
///////////////////////////////////////////////////////////
//                                                       //
//  DogsLife Tic-Tac-Toe Game                            //
//  Copyright (C) 2003 by DogsBody & Ratchet Software    //
//  All Rights Reserved                                  //
//                                                       //
//  This is free software and MAY NOT be sold under any  //
//  circumstances, seperately or bundled.                //
//                                                       //
///////////////////////////////////////////////////////////

//
// Tic-Tac-Toe Game for 210/220/310.
//
// User draws a tic-tac-toe grid on a piece of paper that is 
// between 12 & 18 inches (32-48cm) per side.
//
// Next, user moves ball between two opposite corners, touching
// AIBO's head at each (ie: upper-left & lower-right).  If AIBO 
// registers the coordinates and agrees all is well (by nodding), 
// the game starts.  If AIBO doesn't like the board, he'll shake
// his head and quit the game.  While user is moving the board,
// AIBO will use his eyes to indicate good board coordinates.
// The pause LED will flash slowly during calibration.
//
// AIBO decides who goes first.  If AIBO wants to go first, he'll
// indicate by pointing towards himself with both front paws.   
// If user goes first, AIBO points away from himself with both paws.
//
// User is "O" (because ball is round).  AIBO is "X".
//
// During user move, if ball in invalid position (or over previous move):
//   ERS-210 = AIBO's ears flip in down position, plus mad eyes.
//   ERS-220 = Mad eyes, plus red mouth LED illuminates.
//   ERS-310 = Horn turns RED.
//
// AIBO indicates his move with a single paw either high center low, 
// pointing off left, center, or right.  AIBO tracks the board internally
// and currently doesn't actually observe the board except to monitor
// where the user moves the pink ball.
//
// When AIBO thinks someone has one, or the game tied, he'll do a 
// victory dance, a loser gripe, or a harumf type act.
//
// AIBO will immediately start another round (pointing at himself or user
// with both paws).
//
// Press back sensor to quit the game, or say "STOP".
// 
// If AIBO is losing a lot, he'll likely tire of the game and quit himself.
//
// AIBO can play very well (after a little practice).   An experienced 
// player will very likely tie or lose against AIBO after a while
// (unless AIBO messes up).  :-)
//
// -----------------------
//
// AIBO develops skill in tic-tac-toe, and is controlled by a skill probabilty 
// (likelyhood that AIBO won't overlook something).  If AIBO is a novice, 
// various checks on what the user is doing are overlooked, increasing the 
// likelyhood AIBO will lose.  IF an expert, AIBO won't miss a trick often.
//
//   When AIBO loses the skill goes up (ie: "learns" from mistakes).
//   Skill slowly decreases when AIBO not playing tic-tac-toe (1 tick per ten minutes).
//   Minimum skill level is 20%.  Max is 95%.  
//   Each loss increases skill by 10% (learns most from losing)
//   Each tie increases skill by 5% (learns some from a tie)
//   Each win increases skill by 1% (if AIBO good enough to win, why get better?)
//
#define TTT_HUMAN 1
#define TTT_AIBO 10

#define TTT_UL 0
#define TTT_UC 1
#define TTT_UR 2
#define TTT_ML 3
#define TTT_CENTER 4
#define TTT_MR 5
#define TTT_LL 6
#define TTT_LC 7
#define TTT_LR 8

#define DEFAULT_TTT_SKILL 60
#define AVG_SAMPLES 4
#define DIST_VALID_SAMPLES 6

#define TTT_BOARD_EMPTY (ttt_turn==0)
#define TTT_SMART_ENOUGH() (F_RND100()<skillttt) 
#define SKILLTTT behavior[BEHAVIOR_SKILL_TICTACTOE]

#define GO_TTT_WONGAME 2000

#ifdef ERS310
#define TTT_HEAD_TILT() (Head_Tilt_2*2)
#else
#define TTT_HEAD_TILT() Head_Tilt
#endif


#if ERS220
#define INVALID_MOVE_EYES() Face_3 := Eye_L1 := Eye_R1 := Eye_L2 := Eye_R2 := Eye_L3 := Eye_R3 
#define HAPPY_EYES() Eye_L2 := Eye_R2

#elif ERS310
#define INVALID_MOVE_EYES() Horn_O
#define HAPPY_EYES() Horn_G

#else
#define INVALID_MOVE_EYES() Eye_L1 := Eye_R1 := Eye_L2 := Eye_R2 := Eye_L3 := Eye_R3 
#define HAPPY_EYES() Eye_L2 := Eye_R2
#endif



//
// Tic-Tac-Toe Game Entry Point...
//
:G_TicTacToe_Game
  F_TTT_Adjust_Skill(0)

  SET ttt_distvalid 0
  SET ttt trackarray
  SET ttt_pan1 -999
  SET ttt_tilt1 -999
  SET ttt_pan2 -999
  SET ttt_tilt2 -999
  SET last_pan -999
  SET last_tilt -999

  SET game_count 0
  SET win_count 0
  SET lose_count 0
  SET tie_count 0
  RND limit 3 6		// # of games (losses or wins) AIBO wants to play

  // If user touches back (or 310 tail) at any point, AIBO quits game...
#ifdef ERS310
  Tail_U_ON := Tail_D_ON := Tail_R_ON := Tail_L_ON := 0
#else
  Head_ON := 0
  Back_ON := 0
#endif

  PLAY ACTION ttt_intro2
  CALL G_WAITZERO

  // Make sure head sensor (used in calibration is initially zero)...
  while (Head_ON>0)
    Head_ON := 0
    WAIT:200
  wend
  if (!F_TTT_Calibrate()) then
    RET:1
  endif

  aibo_first := F_RND2()

:TTT_NEWROUND
  SET ttt_turn 0	// 0 = game board empty, 9=game board full

  // Initialize the game board...
  for temp 0 9
    ttt[temp] := 0
  next

  // Decode who goes first...
  if (aibo_first) then // aibo first
    aibo_first := FALSE
    PLAY ACTION ttt_me_first
    CALL G_WAITZERO
    GO TTT_AIBOMOVE
  else
    aibo_first := TRUE
    PLAY ACTION ttt_you_first
    CALL G_WAITZERO
  endif

  #ifdef ERS310
    if (Tail_U_ON || Tail_D_ON || Tail_R_ON || Tail_L_ON) || F_NEED2REST() G_RPS_ENDGAME
  #else
    if (Back_ON>0) || F_NEED2REST() G_RPS_ENDGAME
  #endif
 
:TTT_USERMOVE
  switch (F_TTT_User_Move())
    CASE:GO_ENDGAME GO G_RPS_ENDGAME
    CASE:GO_TTT_WONGAME GO TTT_WONGAME
  ++ttt_turn

  if (temp := F_TTT_EvalGame()) then
    if (temp>=limit) then
      if (tie_count>=limit) G_RPS_TIE_ENDGAME
      if (win_count>lose_count) G_RPS_WON_ENDGAME
      GO G_RPS_LOST_ENDGAME
    endif
    if (game_count>limit*2) then // had enough...
      if (win_count>lose_count) G_RPS_WON_ENDGAME
      if (tie_count>lose_count) G_RPS_TIE_ENDGAME
      GO G_RPS_LOST_ENDGAME
    endif
    GO TTT_NEWROUND
  endif

:TTT_AIBOMOVE
  CALL G_TTT_Aibo_Move
  ++ttt_turn

  if (temp := F_TTT_EvalGame()) then
    if (temp>=limit) then
      if (tie_count>=limit) G_RPS_TIE_ENDGAME
      if (win_count>lose_count) G_RPS_WON_ENDGAME
      GO G_RPS_LOST_ENDGAME
    endif
    if (game_count>limit*2) then // had enough...
      if (win_count>lose_count) G_RPS_WON_ENDGAME
      if (tie_count>lose_count) G_RPS_TIE_ENDGAME
      GO G_RPS_LOST_ENDGAME
    endif
    GO TTT_NEWROUND
  endif
  GO TTT_USERMOVE

:TTT_WONGAME
  F_TTT_Adjust_Skill(1)
  CALL G_RPS_WON_GAME
  if (win_count>=limit) then
    if (tie_count>=limit) G_RPS_TIE_ENDGAME
    if (win_count>lose_count) G_RPS_WON_ENDGAME
    GO G_RPS_LOST_ENDGAME
  endif
  GO TTT_NEWROUND


#ifdef DEBUG
:G_Dump_Board
  PRINT "   %x %x %x" ttt[0] ttt[1] ttt[2]
  PRINT "   %x %x %x" ttt[3] ttt[4] ttt[5]
  PRINT "   %x %x %x" ttt[6] ttt[7] ttt[8]
  RET:1
#endif



///////////////////////////////////////////////////////////////////////////////////
//
//  Evaluate game after last move.
//
//  Returns number of games won or lost if game over.
//
:F_TTT_EvalGame
  LOCAL winpos -1

  // If we won, get excited...
  if (winpos := F_TTT_Win(TTT_AIBO)) then
    ++game_count
/*
    SWITCH (winpos)
      CASE:1 PLAY ACTION ttt_win_top
      CASE:2 PLAY ACTION ttt_win_horzcenter
      CASE:3 PLAY ACTION ttt_win_bottom
      CASE:4 PLAY ACTION ttt_win_left
      CASE:5 PLAY ACTION ttt_win_vertcenter
      CASE:6 PLAY ACTION ttt_win_right
      CASE:7 PLAY ACTION ttt_win_diagonal1
      CASE:8 PLAY ACTION ttt_win_diagonal2
    CALL G_WAITZERO
*/
    F_TTT_Adjust_Skill(1)
    CALL G_RPS_WON_GAME
    if (win_count>2) then // this is now our favorate game
      behavior[BEHAVIOR_FAV_GAME] := 1
    endif
    return win_count
  endif

  // If user won, get upset...
  if (F_TTT_Win(TTT_HUMAN)) then
    ++game_count
    F_TTT_Adjust_Skill(10)
    CALL G_RPS_LOST_GAME
    return lose_count
  endif

  // If tie game, then oh well...
  if (ttt_turn>8) then 
    ++game_count
    F_TTT_Adjust_Skill(5)
    CALL G_RPS_TIE_GAME
    return tie_count
  endif

  return:0



///////////////////////////////////////////////////////////////////////////////////
//
//  Adjust tic-tac-toe skill level (keeping with valid range)...
//
//  Example: 
//    F_TTT_Adjust_Skill(delta)
//
:F_TTT_Adjust_Skill
  ARG:delta

  // If skill initialally invalid, then setup to default...
  skillttt := behavior[BEHAVIOR_SKILL_TICTACTOE]
  if (skillttt<MIN_TTT_SKILL) || (skillttt>MAX_TTT_SKILL) then
    temp := DEFAULT_TTT_SKILL
  endif

  // Adjust & keep within limits...
  skillttt += delta
  if (skillttt<MIN_TTT_SKILL) then
    skillttt := MIN_TTT_SKILL
  endif
  if (skillttt>MAX_TTT_SKILL) then
    skillttt := MAX_TTT_SKILL
  endif
  behavior[BEHAVIOR_SKILL_TICTACTOE] := skillttt
  return:0



///////////////////////////////////////////////////////////////////////////////////
//
//  Calibrate to users tic-tac-toe board...
//  
//  Returns:
//    TRUE if good calibration sampled.
//    FALSE if not.
//
:F_TTT_Calibrate
  LOCAL cstate 0
  LOCAL temp 

  Warn := 0
  Head_ON := 0

  Clock := 0
  timeout := 120 	// set timeout interval for 30 seconds
  timetick := 60	// time of first "tick"
  tickcount := 0

#ifdef ERS310
  PLAY ACTION MOVE.HEAD.FAST 0 -70
  WAIT
#endif

  while (TRUE)
    WAIT:100

    // User wants to abort?
#ifdef ERS310
    if (Tail_U_ON || Tail_D_ON || Tail_R_ON || Tail_L_ON) || F_NEED2REST() then
      Warn := Tail_U_ON := Tail_D_ON := Tail_R_ON := Tail_L_ON := 0
      CALL G_WAITZERO
      CALL G_RPS_ENDGAME
      return FALSE
    endif
#else
    if (Back_ON>0) || F_NEED2REST() then
      Warn := Back_ON := 0
      CALL G_WAITZERO
      CALL G_RPS_ENDGAME
      return FALSE
    endif
#endif

    // If user taking too long (AIBO waits 30 seconds & then gives up)...
    if (Clock>7) then
      Clock := 0
      ++tickcount
      if (F_TTT_User_Timeout()) then
        Warn := 0
        CALL G_WAITZERO
        CALL G_RPS_LOST_ENDGAME
        return FALSE
      endif
    endif

    if (Pink_Ball>0) && (Wait==0) then
      PLAY ACTION TRACK_HEAD PINK_BALL
    endif

    G_TTT_GetBallPos()

    // Waiting for first headpress?
    if (cstate==0) then 
      if (tickcount>1) then
        Warn := !Warn
        HAPPY_EYES() := Pink_Ball
        tickcount := 0
      endif
    else // If not, then waiting for second headpress...
      if (tickcount>0) then // flash faster 
        Warn := !Warn
        HAPPY_EYES() := (F_ABS(pan-ttt_pan1)>20) && (F_ABS(tilt-ttt_tilt1)>20) && ((ttt_pan1>0) && (pan<0)) || ((ttt_pan1<0) && (pan>0))
        tickcount := 0
      endif
    endif

    if (Pink_Ball>0) && (Wait>0) then
      last_pan := Head_Pan+Pink_Ball_Horz
      last_tilt := TTT_HEAD_TILT()+Pink_Ball_Vert
    endif

    // Corner tracking state machine...
    if (cstate>10) then			// Wait for second head-press...
      if (Head_ON>0) && (Pink_Ball>0) && (Wait>0) && (ttt_distvalid>1) then
        PRINT "Calibration Second Sample"
	PLAY ACTION ttt_calib
	WAIT:500
        ttt_pan2 := pan
        ttt_tilt2 := tilt

	// Turn off LED's...
        SET Warn 0			
	HAPPY_EYES() := 0

	// Turn off ball tracking...
        PLAY ACTION MOVE.HEAD.FAST ttt_pan2 ttt_tilt2 
        CALL G_WAITZERO

	// Switch corner position sames if pan2<pan1...
	if (ttt_pan2<ttt_pan1) then
	  SET temp ttt_pan2
	  SET ttt_pan2 ttt_pan1
	  SET ttt_pan1 temp
        endif
        if (ttt_tilt2<ttt_tilt1) then
	  SET temp ttt_tilt2
	  SET ttt_tilt2 ttt_tilt1
	  SET ttt_tilt1 temp
	endif

	if ((ttt_pan2-ttt_pan1)>20) && ((ttt_tilt2-ttt_tilt1)>20) then
          // Now adjust pan settings to the cutoff lines of the tic-tac-toe "#" grid
          temp := F_ABS(ttt_pan1)/6
	  ttt_pan1 += temp
	  ttt_pan2 -= temp
	  ttt_tilt1 += F_ABS(ttt_tilt1)/10
	  ttt_tilt2 -= F_ABS(ttt_tilt2)/15
	  return TRUE
	endif

	return FALSE
      endif
    endif

    if (cstate==10) then		// 1 second after first head-press, reset Head_ON...
      Head_ON := 0
      cstate := 11
    endif
      
    if (cstate>0) && (cstate<10) then 	// After first head-press, wait 500ms...
      ++cstate
    endif

    if (cstate==0) then			// Wait for first head press...
      if (Head_ON>0) && (Pink_Ball>0) && (Wait>0) && (ttt_distvalid>1) then
        PRINT "Calibration First Sample"
	PLAY ACTION ttt_calib
	WAIT:500
        cstate := 1
        ttt_pan1 := pan
        ttt_tilt1 := tilt
        timeout := 120 	// reset timeout interval
        timetick := 60
      endif
    endif
  wend
  return:FALSE



///////////////////////////////////////////////////////////////////////////////////
//
//  Process user's move...
//
//  Returns:
//    GO_ENDGAME      User wants to quit
//    GO_TTT_WONGAME  User resigns
//    GO_IDLE         User made valid move
//
:F_TTT_User_Move
  LOCAL usermove -1
  LOCAL result GO_IDLE

  timeout := 120 	// set timeout interval for 30 seconds
  timetick := 60	// time of first "tick"

  last_usermove := -999
  Head_ON := 0
  Jaw_ON := 0

  if (last_pan>-999) then
    PLAY ACTION MOVE.HEAD.FAST last_pan last_tilt
    CALL G_WAITZERO
    WAIT:100
    if (Pink_Ball==0) then
      PLAY ACTION MOVE.HEAD.FAST 0 (ttt_tilt1+ttt_tilt2)/2
      CALL G_WAITZERO
    endif
  endif

  ttt[20] := -999
  ttt[21] := -999
  ttt[22] := -999
  ttt[23] := -999

  Clock := 0

:TTT_USERLOOP
  repeat
    // If user taking too long (AIBO waits 30 seconds & then gives up)...
    if (Clock>8) then
      Clock := 0
      if (F_TTT_User_Timeout()) TTT_ENDGAME
    endif

    WAIT:100
    if (Pink_Ball>0) && (Wait==0) then
      PLAY ACTION TRACK_HEAD PINK_BALL
    endif
 
    if (Pink_Ball==0) then
      // Turn off eye's if can't see ball...
      if (usermove>=0) then
        INVALID_MOVE_EYES() := 0
      endif
      usermove := -1
    else
      usermove := F_TTT_Compute_Square()

      if (usermove>=0) then
	// Last three samples must be idential (helps avoid user getting burned
        // by a last second head twitch)
	ttt[20] := ttt[21]
	ttt[21] := ttt[22]
	ttt[22] := usermove
      endif

      if (usermove>=0) && (usermove!=last_usermove) then
        last_usermove := usermove
        INVALID_MOVE_EYES() := (ttt[usermove]!=0)

        #if ERS210 | ERS220
        if (ttt[usermove]==0) then
          SWITCH usermove
            CASE:0 SET Eye_L3 1
            CASE:1 SET Eye_L3 1
            CASE:1 SET Eye_R3 1
            CASE:2 SET Eye_R3 1
            CASE:3 SET Eye_L2 1
            CASE:4 SET Eye_L2 1
            CASE:4 SET Eye_R2 1
            CASE:5 SET Eye_R2 1
            CASE:6 SET Eye_L1 1
            CASE:7 SET Eye_L1 1
            CASE:7 SET Eye_R1 1
            CASE:8 SET Eye_R1 1
        endif
        #endif

        PRINT "ballpos=%d\n" usermove
      endif

      // Remember where ball was last...
      if (Pink_Ball>0) && (Wait>0) then
        last_pan := Head_Pan+Pink_Ball_Horz
        last_tilt := TTT_HEAD_TILT()+Pink_Ball_Vert
      endif
    endif

    // Check for voice command...
    CALL G_RPS_ATTENTION
      CASE:GO_VOICE GO TTT_VOICE
      CASE:GO_ENDGAME GO TTT_ENDGAME
      CASE:GO_PRAISE CALL G_PRAISE
      CASE:GO_SCOLD CALL G_SCOLD

    // User wants to quit game?
#ifdef ERS310
    if (Tail_U_ON || Tail_D_ON || Tail_R_ON || Tail_L_ON) || F_NEED2REST() TTT_ENDGAME
#else
    if (Back_ON>0) || F_NEED2REST() TTT_ENDGAME
#endif
 
    // Scratching chin makes AIBO happy...
    if (Jaw_ON>0) then
      CALL G_PRAISE
      WAIT:500
      Jaw_ON := 0
    endif

    // Majority vote of last three samples.  If don't match, or sample not valid yet, 
    // or can't see ball, then don't proceed...
    if (ttt[20]!=ttt[21]) || (ttt[21]!=ttt[22]) || (ttt_distvalid<DIST_VALID_SAMPLES) || (Pink_Ball==0) then
      Head_ON := 0
      usermove := -1
    else
      // If user in illegal position, prevent from playing there...
      if (usermove>=0) then
        if (ttt[usermove]!=0) then
          usermove := -1
        endif
      endif

      // If user touches head & not valid move, then go "huh?"...
      if (Head_ON>0) && (usermove<0) then
        SET init_wait Wait
        PLAY ACTION rps_huh
        CALL G_WAIT
      endif
    endif
  until (Head_ON>0) && (usermove>=0)

  ttt[usermove] := TTT_HUMAN

  SET init_wait Wait
  PLAY ACTION ttt_calib
  CALL G_WAIT

#ifdef DEBUG
  PRINT "User square = %d" usermove
  CALL G_Dump_Board
#endif
  result := GO_IDLE


:TTT_USERDONE
  // Turn off LED's...
  INVALID_MOVE_EYES() := 0

  // Turn off ball tracking...
  PLAY ACTION MOVE.HEAD.FAST Head_Pan TTT_HEAD_TILT()
  CALL G_WAITZERO
  return:result


:TTT_VOICE
  SWITCH VoiceCmd
    CASE:VOICE_STOP GO TTT_ENDGAME		// Stop
    CASE:VOICE_FINDBALL GO TTT_DOVOICE		// Find Ball
    CASE:VOICE_KICKBALL GO TTT_DOVOICE		// Kick Ball
    CASE:VOICE2_PINKBALL GO TTT_DOVOICE		// Pink Ball
    CASE:VOICE2_RIGHTKICK GO TTT_DOVOICE	// Right Kick - ZZZ - Should add right kick code
    CASE:VOICE2_LEFTKICK GO TTT_DOVOICE		// Left Kick - ZZZ - Should add left kick code
    CASE:VOICE2_RIGHTTOUCH GO TTT_DOVOICE	// Right Touch - ZZZ - Should add right touch code
    CASE:VOICE2_LEFTTOUCH GO TTT_DOVOICE	// Left Touch - ZZZ - Should add left touch code
    CASE:VOICE_LAYDOWN GO TTT_DOVOICE		// Lie Down / Play Dead
    CASE:VOICE_GOODNIGHT GO TTT_DOVOICE		// Good Night
    CASE:VOICE_GOODBYE GO TTT_DOVOICE		// Goodbye
    CASE:VOICE2_SEEYOU GO TTT_DOVOICE		// See You
    CASE:VOICE2_YOUWON GO TTT_USER_RESIGNED	// Win!
    CASE:VOICE2_YOULOST GO TTT_SAYWHAT 		// Lose?  I don't think so..
  GO TTT_USERLOOP

:TTT_DOVOICE
  PRINT "Quit Tic-Tac-Toe.  VoiceCmd=%d" VoiceCmd
  SET AP_Voice_Cmd (VoiceCmd-base_voice_cmd)	// Make normal loop respond to voice command...
  SET AP_Voice_Level MIN_VOICE_LEVEL
  SET result GO_ENDGAME
  GO TTT_USERDONE

:TTT_ENDGAME
  SET result GO_ENDGAME
  GO TTT_USERDONE

:TTT_USER_RESIGNED
  SET result GO_TTT_WONGAME
  GO TTT_USERDONE

:TTT_SAYWHAT
  PLAY ACTION rps_huh
  CALL G_WAITZERO
  GO TTT_USERLOOP
 


///////////////////////////////////////////////////////////////////////////////////
//
//  Detect if user taking too long to make a move...
//
:F_TTT_User_Timeout
  if (timeout>0) then
    --timeout
    if (timeout==timetick) then
      if (timeout>28) then
        timetick -= 8
      else 
        if (timeout>12) then
          timetick -= 4
        else
          if (timeout>4) then
            timetick -= 2
          else
            timetick -= 1
          endif
        endif
      endif
      if (timetick>0) then
        PLAY ACTION ttt_timetick
      else
        PLAY ACTION ttt_timeout
      endif
    endif
  endif
  return:(timeout<1)



///////////////////////////////////////////////////////////////////////////////////
//
//  Get ball position (tilt & pan), with compensation for head sticking or jitter.  
//  Since Pink_Ball_Dist is a useful parameter, but wildly unstable, an average 
//  of the last four samples is used.
//
//  Returns:
//    "tilt" and "pan"
//
:G_TTT_GetBallPos
  LOCAL average_dist 0
  if (Pink_Ball==0) then
    ttt_distvalid := 0
    RET:1
  endif
  if (++ttt_distvalid > DIST_VALID_SAMPLES) then
    ttt_distvalid := DIST_VALID_SAMPLES
  endif
  for i 1 AVG_SAMPLES-1
    average_dist += (ttt[10+i] := ttt[11+i])
  next
  WAIT:64
  dist := (average_dist+(ttt[10+AVG_SAMPLES]:=Pink_Ball_Dist))/(AVG_SAMPLES*50)
  tilt := TTT_HEAD_TILT() + Pink_Ball_Vert + dist
  pan := Head_Pan + Pink_Ball_Horz
  PRINT "TTT BallPos.  dist=%d  tilt=%d  pan=%d" dist tilt pan
  RET:1



///////////////////////////////////////////////////////////////////////////////////
//
//  Base on head position, compute square AIBO is looking at.  Calibration must
//  have been performed prior to calling this function.
//
//  Returns:
//    0..8 = Square closest to what AIBO is looking at.
//    -1 if ball not visible
//
:F_TTT_Compute_Square
  LOCAL rowbase 0
  LOCAL panlimit 0

  if (Pink_Ball==0) then
    ttt_distvalid := 0
    return -1
  endif

  CALL G_TTT_GetBallPos

  SET rowbase 3
  if (tilt < ttt_tilt1) then
    SET rowbase 6
  else 
    if (tilt > ttt_tilt2) then
      SET rowbase 0
    endif
  endif

  panlimit := F_ABS(ttt_pan1)
  if (F_ABS(ttt_pan2)<panlimit) then
    panlimit := F_ABS(ttt_pan2)
  endif
  panlimit /= 2
  if (pan>panlimit) then
    return rowbase
  endif
  if (pan<-panlimit) then
    return rowbase+2
  endif
  return rowbase+1



///////////////////////////////////////////////////////////////////////////////////
//
//  Periodically flash eye to indicating AIBO is thinking...
//
:G_TTT_Thinking
  if (Clock>63) && (Wait==0) then
    Clock := 0
    PLAY ACTION ttt_thinking
  endif
  RET:1



///////////////////////////////////////////////////////////////////////////////////
//
//  Process AIBO's move...
//
:G_TTT_Aibo_Move
  LOCAL aibo_move
  aibo_move := F_TTT_PickMove()
  CALL G_WAITZERO // finish any skits started while thinking...

  if (aibo_move>=0) then
    ttt[aibo_move] := TTT_AIBO
  endif

#ifdef DEBUG
  PRINT "Aibo square = %d" aibo_move
  CALL G_Dump_Board
#endif

  switch (aibo_move)
    case:TTT_UL PLAY ACTION ttt_pick_upperleft
    case:TTT_UC PLAY ACTION ttt_pick_uppercenter
    case:TTT_UR PLAY ACTION ttt_pick_upperright
    case:TTT_ML PLAY ACTION ttt_pick_centerleft
    case:TTT_CENTER PLAY ACTION ttt_pick_centersquare
    case:TTT_MR PLAY ACTION ttt_pick_centerright
    case:TTT_LL PLAY ACTION ttt_pick_lowerleft
    case:TTT_LC PLAY ACTION ttt_pick_lowercenter
    case:TTT_LR PLAY ACTION ttt_pick_lowerright
  GO G_WAITZERO




///////////////////////////////////////////////////////////////////////////////////
//
//  Pick AIBO's move...
//
//  Returns: 
//    Game grid coordinate of next move, or -1 if none (ie: board full).
//
:F_TTT_PickMove
  LOCAL i
  LOCAL j
  LOCAL k
  LOCAL near_win
  LOCAL possible_win

  // Board full?
  if (ttt_turn>8) then
    return -1
  endif

  // First, see if AIBO can win...
  if (F_RND2()) then
    for i 0 2
      if ((j:=F_TTT_CheckHorz(i,TTT_AIBO))>=0) then
        return j
      endif
      if ((j:=F_TTT_CheckVert(i,TTT_AIBO))>=0) then
        return j
      endif
    next
  else // look bottom-up first instead of top-down (for variety)...
    for i 0 2
      if ((j:=F_TTT_CheckHorz(2-i,TTT_AIBO))>=0) then
        return j
      endif
      if ((j:=F_TTT_CheckVert(2-i,TTT_AIBO))>=0) then
        return j
      endif
    next
  endif
  if ((j:=F_TTT_CheckDiagonal(TTT_AIBO))>=0) then
    return j
  endif

  // Check if AIBO in trouble.  If about to lose, block user...
  if (F_RND2()) then
    for i 0 2
      if ((j:=F_TTT_CheckHorz(i,TTT_HUMAN))>=0) then
        return j
      endif
      if ((j:=F_TTT_CheckVert(i,TTT_HUMAN))>=0) then
        return j
      endif
    next
  else // look bottom-up first instead of top-down (for variety)...
    for i 0 2
      if ((j:=F_TTT_CheckHorz(2-i,TTT_HUMAN))>=0) then
        return j
      endif
      if ((j:=F_TTT_CheckVert(2-i,TTT_HUMAN))>=0) then
        return j
      endif
    next
  endif
  if ((j:=F_TTT_CheckDiagonal(TTT_HUMAN))>=0) then
    return j
  endif

  /*** Neither user or AIBO about win, so time for strategy... ***/

  // If first move...
  if (ttt_turn==0) then
    if (TTT_SMART_ENOUGH()) then // pick corner or center
      RND rndnum 0 4
      return (rndnum*2)
    endif
    // Otherwise, just pick anything randomly...
    RND rndnum TTT_UL TTT_LR
    return rndnum
  endif

  // If center square available, grab it...
  if (ttt_turn<=2) && (TTT_SMART_ENOUGH() && (ttt[TTT_CENTER]==0)) then
    if (ttt_turn==1) then
      return TTT_CENTER
    endif

    // AIBO's first move was in corner, so center is a good choice no matter what...
    if (ttt[TTT_UL]==TTT_AIBO) || (ttt[TTT_UR]==TTT_AIBO) || (ttt[TTT_LL]==TTT_AIBO) || (ttt[TTT_LR]==TTT_AIBO) then
      return TTT_CENTER
    endif

    // If AIBO's first move was top or bottom center, and far side clear, then center ok...
    if ((ttt[TTT_UC]==TTT_AIBO) && (ttt[TTT_LC]==0)) || ((ttt[TTT_LC]==TTT_AIBO) && (ttt[TTT_UC]==0)) then
      return TTT_CENTER
    endif

    // If AIBO's first move was left or right center, and far side clear, then center ok...
    if ((ttt[TTT_ML]==TTT_AIBO) && (ttt[TTT_MR]==0)) || ((ttt[TTT_MR]==TTT_AIBO) && (ttt[TTT_ML]==0)) then
      return TTT_CENTER
    endif
  endif

  //
  // Look for following if after second user move board looks like:
  //     o| |	 
  //     -+-+-   x = aibo
  //      |x|    o = user
  //     -+-+-
  //      | |o
  //
  // If found, choose a edge-center square to force user to defend...
  //
  if (ttt_turn==3) && (ttt[TTT_CENTER]==TTT_AIBO) then
    i := (ttt[TTT_UL]==TTT_HUMAN) && (ttt[TTT_LR]==TTT_HUMAN)
    j := (ttt[TTT_UR]==TTT_HUMAN) && (ttt[TTT_LL]==TTT_HUMAN)
    if (i || j) then
      if (TTT_SMART_ENOUGH()) then // pick an edge-center square
	repeat
          RND rndnum 0 3
	  rndnum := (rndnum*2)+1
	until ttt[rndnum]==0
	return rndnum
      endif
      // otherwise, pick a corner...
      if (F_RND100()<50) then // top-row
	return TTT_UL+2*i
      else // bottom-row
	return TTT_LL+2*j
      endif
    endif
  endif
 
  // Prevent user from setting up two "near" win's...
  for i 0 9
    if (ttt[i]==0) && TTT_SMART_ENOUGH() then
      k := 0
      for j 0 9
        if (i!=j) && (ttt[j]==0) then
          CALL G_TTT_Thinking
          ttt[i] := TTT_HUMAN
	  ttt[j] := TTT_HUMAN
 	  k += (F_TTT_Win(TTT_HUMAN)>0)
	  ttt[i] := 0
	  ttt[j] := 0
	  if (k>1) then
	    return i
	  endif
        endif
      next
    endif
  next

  // Look for situations where AIBO gets two "near" win's...
  near_win := -1
  possible_win := -1
  for i 0 9
    if (ttt[i]==0) && TTT_SMART_ENOUGH() then
      k := 0
      for j 0 9
        if (i!=j) && (ttt[j]==0) then
          CALL G_TTT_Thinking
	  ttt[i] := TTT_AIBO
	  ttt[j] := TTT_AIBO
 	  if (F_TTT_Win(TTT_AIBO)>0) then // possible AIBO win
            k++ // inc near win count (assuming first AIBO move at 'i', and second at 'j')
	    if (possible_win<0) || (F_RND100()>50) then // remember any possible near win
	      possible_win := i
	    endif
	  endif
	  ttt[i] := 0
	  ttt[j] := 0
	  if (k>1) && ((near_win<0) || (F_RND100()>50)) then
	    near_win := i
	  endif
        endif
      next
    endif
  next

  // If there is a way to get two possible winning moves, go for it...
  if (near_win>=0) then
    return near_win
  endif

  // If there is any way to win in two moves, go for it...
  if (possible_win>=0) then
    return possible_win
  endif

  // Pick something at random...
  repeat
    RND rndnum TTT_UL TTT_LR 
  until ttt[rndnum]==0
  return rndnum



///////////////////////////////////////////////////////////////////////////////////
//
//  Check for impending horizontal win...
//
//  Example: 
//    F_TTT_CheckHorz(row,player)
//
//  Returns: 
//    Game grid coordinate that can block impending win, or -1 if none.
//
:F_TTT_CheckHorz
  ARG row
  ARG player
  LOCAL j
  j := row*3
  CALL G_TTT_Thinking
  if (ttt[j]==player) && (ttt[j+1]==player) && (ttt[j+2]==0) then // XX- horz
    return j+2
  endif
  if (ttt[j]==player) && (ttt[j+2]==player) && (ttt[j+1]==0) then // X-X horz
    return j+1
  endif
  if (ttt[j+1]==player) && (ttt[j+2]==player) && (ttt[j]==0) then // -XX horz
    return j
  endif
  return -1



///////////////////////////////////////////////////////////////////////////////////
//
//  Check for impending vertical win...
//
//  Example: 
//    F_TTT_CheckVert(column,player)
//
//  Returns: 
//    Game grid coordinate that can block impending win, or -1 if none.
//
:F_TTT_CheckVert
  ARG col
  ARG player
  CALL G_TTT_Thinking
  if (ttt[col]==player) && (ttt[col+3]==player) && (ttt[col+6]==0) then // XX- vert
    return col+6
  endif
  if (ttt[col]==player) && (ttt[col+3]==0) && (ttt[col+6]==player) then // X-X vert
    return col+3
  endif
  if (ttt[col]==0) && (ttt[col+3]==player) && (ttt[col+6]==player) then // -XX vert
    return col
  endif
  return -1



///////////////////////////////////////////////////////////////////////////////////
//
//  Check for impending diagonal win...
//
//  Example: 
//    F_TTT_CheckDiagonal(player)
//
//  Returns: 
//    Game grid coordinate that can block impending win, or -1 if none.
//
:F_TTT_CheckDiagonal
  ARG player
  CALL G_TTT_Thinking
  if (ttt[TTT_UL]==player) && (ttt[TTT_CENTER]==player) && (ttt[TTT_LR]==0) then  // XX- diagonal
    return TTT_LR
  endif
  if (ttt[TTT_UL]==player) && (ttt[TTT_CENTER]==0) && (ttt[TTT_LR]==player) then  // X-X diagonal
    return TTT_CENTER
  endif
  if (ttt[TTT_UL]==0) && (ttt[TTT_CENTER]==player) && (ttt[TTT_LR]==player) then  // -XX diagonal
    return TTT_UL
  endif

  if (ttt[TTT_UR]==player) && (ttt[TTT_CENTER]==player) && (ttt[TTT_LL]==0) then  // XX- diagonal
    return TTT_LL
  endif
  if (ttt[TTT_UR]==player) && (ttt[TTT_CENTER]==0) && (ttt[TTT_LL]==player) then  // X-X diagonal
    return TTT_CENTER
  endif
  if (ttt[TTT_UR]==0) && (ttt[TTT_CENTER]==player) && (ttt[TTT_LL]==player) then  // -XX diagonal
    return TTT_UR
  endif
  return -1



///////////////////////////////////////////////////////////////////////////////////
//
//  See if someone has won...
//
//  Example: 
//    F_TTT_Win(player)
//
//  Return:
//    0 = No winner
//    1 = top horz
//    2 = middle horz
//    3 = bottom horz
//    4 = left vert
//    5 = center vert
//    6 = right vert
//    7 = top-left to bottom-right diagonal
//    8 = top-right to bottom-left diagonal
//
:F_TTT_Win
  ARG player
  LOCAL i 0
  LOCAL j 0
  for i 0 2
    if (ttt[j]==player) && (ttt[j+1]==player) && (ttt[j+2]==player) then
      return (i+1) // Horz win
    endif
    if (ttt[i]==player) && (ttt[i+3]==player) && (ttt[i+6]==player) then
      return (i+4) // Vert win
    endif
    ADD j 3
  next
  // Diagonal win?
  if (ttt[TTT_UL]==player) && (ttt[TTT_CENTER]==player) && (ttt[TTT_LR]==player) then
    return 7
  endif
  if (ttt[TTT_UR]==player) && (ttt[TTT_CENTER]==player) && (ttt[TTT_LL]==player) then
    return 8
  endif
  return:0

