
#define FOCUS_SPEED 500
#define PINK_THRESHOLD 10
#define BLUE_THRESHOLD 10
#define GREEN_THRESHOLD 5
#define MAX_THRESHOLD 400

#define MIN_BALL_DIST 550
#define IDEAL_BALL_DIST 700
#define MAX_BALL_DIST 800


//
// Look for charging station & attempt to self-dock.  We won't be able to get
// up (because Sony broke the charger mode), but self docking at least gets us
// half-way there.
//
:G_SELFDOCK
  if (AiboType == 310) then
    RET:GO_IDLE
  endif

  // Load initial direction to search.  This is helpful in situations where the
  // charger is in a corner.  Knowing the correct direction to search helps AIBO
  // avoid walking into a wall...
  VLOAD dockdir // -1=clockwise.  1=counter clockwise
  if (F_ABS(dockdir) != 1) then
    SET dockdir 1
  endif

  SET sd_scancount 0
  SET sd_reversecount 0
  SET sd_dockcount 0
  SET sd_colorscan FALSE	// Indicates color self-calibration completed
  SET Head_ON 0
  SET Back_ON 0

  // Make sure AIBO standing before start looking...
  if (!POSE_STANDING()) then
    PLAY ACTION STAND
    CALL G_WAITZERO
  endif


//
// Step 1.  Find the ball.  Must be at shoulder height.  If there is a loose
// ball next to the charging color tower, this probably won't work too well...
//
:SD_FINDBALL
  sd_slowsearch := 4

  CALL GSD_FOCUS_BALL
  for i 1 4
    // If battery gets too low, then shutdown now...
    if (Back_ON>0) || F_NEED2REST() then
      RET:GO_RESTPOS
    endif

    PLAY ACTION MOVE.HEAD.FAST 0 0
    CALL G_WAITZERO
    if (Pink_Ball>0) then
      CALL GSD_TRACKBALL
     GO SD_SEEBALL
    endif

    PLAY ACTION MOVE.HEAD.SLOW 90*dockdir 0
    if (FSD_WAITSCAN()==GO_SEEBALL) SD_SEEBALL

    PLAY ACTION MOVE.HEAD.SLOW -90*dockdir 0
    if (FSD_WAITSCAN()==GO_SEEBALL) SD_SEEBALL

    PLAY ACTION MOVE.HEAD.FAST 0 0
    PLAY ACTION TURN 100
    CALL G_WAITZERO
  next


//
// If don't see ball using normal ball detection, lets try it manually.
// This is a long shot, but might allow AIBO to find the charger from a
// much greater distance (6 to 8 feet, vs 3-4 feet).
//
:SD_BALLSEARCH
  for i 1 7
    SWITCH i
      CASE 1 : PLAY ACTION MOVE.HEAD.FAST 90 0 
      CASE 2 : PLAY ACTION MOVE.HEAD.FAST 60 0 
      CASE 3 : PLAY ACTION MOVE.HEAD.FAST 30 0 
      CASE 4 : PLAY ACTION MOVE.HEAD.FAST 0 0 
      CASE 5 : PLAY ACTION MOVE.HEAD.FAST -30 0 
      CASE 6 : PLAY ACTION MOVE.HEAD.FAST -60 0 
      CASE 7 : PLAY ACTION MOVE.HEAD.FAST -90 0 
    CALL G_WAITZERO

    if (FSD_TRACKHEAD(COLORMASK_PINK,44,36)==GO_AHEAD) SD_BALLALIGNLOOP
  next

  if (--sd_slowsearch<0) then
    RET:1
  endif

  PLAY ACTION TURN 100
  CALL G_WAITZERO
  GO SD_BALLSEARCH

:SD_BALLALIGNLOOP
  // If battery gets too low, then shutdown now...
  if (Back_ON>0) || F_NEED2REST() then
    PLAY ACTION MOVE.HEAD.FAST Head_Pan Head_Tilt	// turn off tracking
    CALL G_WAITZERO
    RET:GO_RESTPOS
  endif

  // Once AIBO's normal ball detection kicks in, let's get busy...
  if (Pink_Ball>0) && (Pink_Ball_Dist<1000) SD_SEEBALL

  if (FSD_TRACKHEAD(COLORMASK_PINK,44,36)==GO_IDLE) SD_BALLSEARCH

  CSET:Head_Pan > 10:10
  CSET:Head_Pan < -10:-10
  CSET:Head_Pan >= -10:0
  if (Context != 0) then
    SET align_hint (Context<0) ? 1 : 2
    PLAY ACTION TURN Context
    CALL G_WAITZERO
    GO SD_BALLALIGNLOOP
  endif

  if (Head_Tilt > -30) then
    SET init_wait Wait
    PLAY ACTION WALK 0 60
    CALL G_WAITZERO
    GO SD_BALLALIGNLOOP
  endif
  GO SD_BALLSEARCH


:SD_NOT_TOWER
  // If saw a ball, but didn't find tower, need color scan again...
  SET sd_colorscan FALSE	
  PLAY ACTION MOVE.HEAD.FAST Head_Pan Head_Tilt
  PLAY ACTION TURN 120
  CALL G_WAITZERO
  GO SD_FINDBALL



//
// Step 2.  Move until ball approx 700mm away (+/- 50).  This is the distance
// from the charger alignment disc to the alignment tower.   If ball near 
// shoulder height (gauged by Head_Tilt being >=-7 & <=7), we have a winner.
// If not, turn away from this ball and search again.  
//
// If pass through this scan more than 3 times, give it up.  We are
// probably just relocking onto a random ball somewhere and cannot see
// the charger from here, or a second ball is too close to the tower.
//
// Code for approaching the ball is similar to the soccer ball tracking logic.
//
:SD_SEEBALL
  if (Pink_Ball==0) then
    PLAY ACTION MOVE.HEAD.FAST Head_Pan Head_Tilt	// turn off tracking
    CALL G_WAITZERO
    GO SD_FINDBALL
  endif

  // If battery gets too low, then shutdown now...
  if (Back_ON>0) || F_NEED2REST() then
    PLAY ACTION MOVE.HEAD.FAST Head_Pan Head_Tilt	// turn off tracking
    CALL G_WAITZERO
    RET:GO_RESTPOS
  endif

  CALL GSD_TRACKBALL
  CALL GSD_GETBALLPOS

  // If head looking too far down, then we havent found the charging tower.
  // Turn off tracking & turn away from wrong ball...
  if ((ball_vert>9) || (ball_vert<-15)) && (ball_horz>-45) && (ball_horz<45) then
    PLAY ACTION MOVE.HEAD.FAST Head_Pan Head_Tilt
    PLAY ACTION TURN 180
    CALL G_WAITZERO
    GO SD_FINDBALL
  endif

  // Keep going if not right distance away...
  if (FSD_MUST_MOVE()) then
    if (F_ABS(Head_Pan)>30) CALL GSD_TURN2BALL
    CALL GSD_MOVE2BALL
    CALL GSD_TURN2BALL
    GO SD_SEEBALL
  endif

  // Should be close enough now.  Turn until lined up...
  CALL GSD_ALIGN2BALL
  if (Pink_Ball==0) SD_FINDBALL



//
// Step 3.  In range of a shoulder height ball.  Confirm we can see
// the green & blue tower support.  If not, this isn't the charger.
// Turn away and look again...
//
:SD_VERIFYTOWER
  CALL GSD_COLORCHECK

  if (pink_level>=PINK_THRESHOLD) && (blue_level>=BLUE_THRESHOLD) then
    if (F_ABS(pink_x-blue_x)>16) || (pink_y>=blue_y) SD_NOT_TOWER
  else 
    if (pink_level>=PINK_THRESHOLD) && (green_level>=GREEN_THRESHOLD) then
      if (F_ABS(pink_x-green_x)>16) || (pink_y>=green_y) SD_NOT_TOWER
    else
      GO SD_NOT_TOWER
    endif
  endif

  // Who hoo!  Found the tower, lets boogie...
  if (mood_happy>4) && (F_RND100()<5) CALL G_HAPPYDANCE


//
// Step 4.  We found the charger.  Woo hoo!!!
//
// Now look for the alignment disc at our feet.  If lucky, we'll 
// find it there.  If not, then start searching for it by walking 
// in a circle around the charging tower (at the 700mm radius).  
//
// If run into something, turn around and walk in the opposite 
// direction.  Owner might have tucked the station against a wall
// (a good idea actually, since the thing is a tripping hazard).
//
// If run into second something, give up...
//
:SD_FINDDISC
  if (dockdir>0) then	// look left to right
    for i 1 6
      SWITCH i
        CASE:1 PLAY ACTION MOVE.HEAD.FAST -50 -20	// look to left
        CASE:2 PLAY ACTION MOVE.HEAD.FAST -35 -45	// look to left
        CASE:3 PLAY ACTION MOVE.HEAD.FAST 0 -90		// look in front
        CASE:4 PLAY ACTION MOVE.HEAD.FAST 35 -45	// look to right
        CASE:5 PLAY ACTION MOVE.HEAD.FAST 50 -20	// look to right
        CASE:6 PLAY ACTION MOVE.HEAD.FAST 0 -65		// look in front again
      CALL G_WAITZERO
      WAIT:200
      if (FSD_SEE_DISC()) SD_ALIGN
    next
  else 			// look right to left
    for i 1 6
      SWITCH i
        CASE:1 PLAY ACTION MOVE.HEAD.FAST 50 -20	// look to right
        CASE:2 PLAY ACTION MOVE.HEAD.FAST 35 -45	// look to right
        CASE:3 PLAY ACTION MOVE.HEAD.FAST 0 -90		// look in front
        CASE:4 PLAY ACTION MOVE.HEAD.FAST -35 -45	// look to left
        CASE:5 PLAY ACTION MOVE.HEAD.FAST -50 -20	// look to left
        CASE:6 PLAY ACTION MOVE.HEAD.FAST 0 -65		// look in front again
      CALL G_WAITZERO
      WAIT:200
      if (FSD_SEE_DISC()) SD_ALIGN
    next
  endif

  if (F_RND100()<10) then
    PLAY ACTION sad_eyes
  endif

  CALL GSD_FOCUS_BALL
  CALL G_WAITZERO


  // Occasionally, get itchy while doing this...
  RND rndnum 1 100
  SWITCH:rndnum
    CASE:1 PLAY ACTION scratch_rear_right
    CASE:2 PLAY ACTION scratch_rear_ear_right
    CASE:3 PLAY ACTION scratch_rear_left
    CASE:4 PLAY ACTION scratch_rear_ear_left
  CALL G_WAITZERO

  // Look at ball & track it while turning so we can get close to 
  // the ideal 60 degree turn...
  PLAY ACTION MOVE.HEAD.FAST 0 0
  CALL G_WAITZERO

:SD_TOWERWALK
  CALL GSD_TRACKBALL
  if (Pink_Ball==0) || (Wait==0) SD_FINDBALL

  // If battery gets too low, then shutdown now...
  if (Back_ON>0) || F_NEED2REST() then
    PLAY ACTION MOVE.HEAD.FAST Head_Pan Head_Tilt	// turn off tracking
    CALL G_WAITZERO
    RET:GO_RESTPOS
  endif

  SET init_wait Wait

  // Backup if too close...
  while (FSD_GETBALLDIST()<IDEAL_BALL_DIST) && (Pink_Ball>0) && (Wait>0)
    if (!FSD_SLOWTURN2BALL(4)) SD_FINDBALL
    PLAY ACTION WALK 180 50
    CALL G_WAIT
  wend

  // Want to turn by 60 degrees.  However, when turning clockwise, AIBO's accuracy
  // is not so great.  On hard floors it adds an extra 20 degrees or so.  Thus
  // a little code to compensate...
  if (dockdir<0) then
    pan := -60
    PLAY ACTION TURN 60
  else 
    pan := 70
    PLAY ACTION TURN -30
  endif

  CALL G_WAIT
  WAIT 1000

  // If not turned enough, turn some more...
  turn_angle := 10*dockdir
  while (Pink_Ball>0) && (Wait>0) && (Head_Pan<50) && (Head_Pan>-55)
    pan += turn_angle
    PLAY ACTION TURN -turn_angle
    CALL G_WAIT
    WAIT:100
  wend

  // If turned too much, turn back...
  while (Pink_Ball>0) && (Wait>0) && ((Head_Pan>70) || (Head_Pan<-75))
    pan -= turn_angle
    PLAY ACTION TURN turn_angle
    CALL G_WAIT
    WAIT:100
  wend

  PLAY ACTION MOVE.HEAD.FAST 0 WALKHEADTILT
  CALL G_WAIT

  PLAY ACTION WALK 0 400
  CALL GSD_WAITWALK
    CASE:GO_STOP GO SD_OBSTACLE

  PLAY ACTION TURN pan
  CALL G_WAIT
  GO SD_FINDBALL

:SD_OBSTACLE
  // Encountered an obstacle - reverse direction...
  PLAY ACTION mad_eyes
  PLAY ACTION TURN pan
  CALL G_WAIT

  MUL dockdir -1
  ++sd_reversecount
  if (sd_reversecount<4) SD_FINDBALL

  // Give it up...  Something is causing us to reverse direction too often.
  // Restore camera settings to AIBO defaults...
  CALL GSD_FOCUS_BALL
  RET:GO_IDLE


:SD_TOWERLOCK
  CALL GSD_ALIGN2BALL
  GO SD_TOWERWALK



//
// Step 5.  Position AIBO aligned straight-on with the disc & tower.   
// Approach with alignment disc off to one side.  
//
// Initially we have to be careful, since at a distance its easily
// possible for AIBO to see both the disc and charger.  Therefore
// while closing some distance, the disc is centered near the top
// of the field of view, or off center left or right.  Once close enough, 
// the disc is centered properly.
//
// Once close to the disc, pan head to 90 and look for the tower ball.  
// If not seen, walk forward & turn.  Realign the head as described 
// above, and repeat.
//
// Once tower ball spotted, enable ball tracking to get a precise fix.
// Continue the walk/turn around disc (looking down then up again)
// until ball pan less than 70 degrees.
//
:SD_ALIGN
  CALL GSD_BLUE_LEVEL

  align_x := 44
  align_y := 15
  aligned_disc := FALSE
  target_pan := 0
  target_side := 0

  if (Head_Pan>=15) then
    target_side := -1
    target_pan := -35
    align_x := 54
    align_y := 20
  endif

  if (Head_Pan<=-15) then
    target_side := 1
    target_pan := 35
    align_x := 34
    align_y := 20
  endif

  lookup_startpos := 0
  align_hint := 0
  align_far := TRUE


:SD_ALIGNLOOP
  CALL GSD_FOCUS_BLUE
  SWITCH:FSD_TRACKHEAD(COLORMASK_BLUE,align_x,align_y)
    CASE:GO_AHEAD GO SD_ALIGNEVAL
    CASE:GO_SEEBALL GO SD_TOWERLOCK

  SWITCH:align_hint
    CASE:0 if (FSD_REAQUIRE_DISC()==GO_IDLE) SD_FINDBALL
    CASE:1 if (FSD_REAQUIRE_LEFT()==GO_IDLE) SD_FINDBALL
    CASE:2 if (FSD_REAQUIRE_RIGHT()==GO_IDLE) SD_FINDBALL
    CASE:3 if (FSD_REAQUIRE_FORWARD()==GO_IDLE) SD_FINDBALL
    CASE:4 if (FSD_REAQUIRE_BACKWARD()==GO_IDLE) SD_FINDBALL

  SWITCH:FSD_TRACKHEAD(COLORMASK_BLUE,align_x,align_y)
    CASE:GO_IDLE GO SD_FINDBALL
    CASE:GO_SEEBALL GO SD_TOWERLOCK

:SD_ALIGNEVAL
  // If battery gets too low, then shutdown now...
  if (Back_ON>0) || F_NEED2REST() then
    RET:GO_RESTPOS
  endif

  // If head angle too high & see pink, then we locked onto the tower & not the disc...
  // Walk around the disc some more & hopefully get closer...
  //
  // FUTURE: Add smarts to try and ignore non-tower ball (ie: normal toy ball
  // that just happens to be next to charger).
  //
  if (Head_Tilt>=-30) then
    CALL GSD_FOCUS_BALL
    if (Pink_Ball>0) SD_TOWERLOCK
    CALL GSD_FOCUS_BLUE
  endif

  // If disc far away, then walk straight at it.  Once close enough, the normal angled
  // approach code is used...
  if (align_far) then
    if (Head_Tilt>=-45) then 
      CSET:Head_Pan > 10:10
      CSET:Head_Pan < -10:-10
      CSET:Head_Pan >= -10:0
      if (Context != 0) then
        SET align_hint (Context<0) ? 1 : 2
        PLAY ACTION TURN Context
	CALL G_WAITZERO
        GO SD_ALIGNLOOP
      endif

      // If disc too close, back off...
      if (Head_Tilt < -90) then
        SET align_hint 3
        SET init_wait Wait
        PLAY ACTION WALK 180 25
	CALL G_WAITZERO
        GO SD_ALIGNLOOP
      endif

      // If disc not close enough, step forward...
      if (Head_Tilt>=-70) then
        SET align_hint 4
        SET init_wait Wait
        PLAY ACTION WALK 0 25
	CALL G_WAITZERO
        GO SD_ALIGNLOOP
      endif
    endif

    // Close enough to use normal angled approach now...
    align_x := 44
    align_y := 36
    align_far := FALSE
  endif

  // If ball far away, so turn straight at it...
  SET turn_angle 0
  if (Head_Tilt>=-30) then 
    CSET:Head_Pan > 10:10
    CSET:Head_Pan < -10:-10
    CSET:Head_Pan >= -10:0
    SET turn_angle Context
  else
    // If disc off to side, turn to face it...
    if (Head_Pan > 30+target_pan) then
      SET turn_angle 20
    endif
    if (Head_Pan > 10+target_pan) then
      SET turn_angle 10
    endif
    if (Head_Pan < -30+target_pan) then
      SET turn_angle -20
    endif
    if (Head_Pan < -10+target_pan) then
      SET turn_angle -10
    endif
  endif

  // Turn as required.  If turn results in loosing track of disc, 
  // attempt to guess where it might have gone & reaquire.  Helps 
  // avoid reverting to a ball search...
  if (turn_angle != 0) then
    PLAY ACTION TURN turn_angle
    CALL G_WAITZERO
    align_hint := (turn_angle<0) ? 1 : 2
    GO SD_ALIGNLOOP
  endif

  // If disc too close, back off...
  if (Head_Tilt < -90) then
    SET align_hint 3
    PLAY ACTION WALK 180 25
    CALL G_WAITZERO
    GO SD_ALIGNLOOP
  endif

  // If disc not at our feet, step closer...
  SWITCH:0
  CSET:Head_Tilt >= -35:75
  CSET:Head_Tilt >= -40:75
  CSET:Head_Tilt >= -45:50
  CSET:Head_Tilt >= -50:50
  CSET:Head_Tilt >= -55:50
  CSET:Head_Tilt >= -60:25
  CSET:Head_Tilt >= -70:25
  if (Context != 0) then
    SET align_hint 4
    PLAY ACTION WALK 0 Context
    CALL G_WAITZERO
    GO SD_ALIGNLOOP
  endif

  // Should now be next to the disc...  We need to walk around the edge 
  // of it until the tower ball can be seen.

  // If alignment disc to our side...
  if (target_side<>0) then
    // See if ball visible...
    ball_dist := -1
    CALL GSD_FOCUS_BALL

    PLAY ACTION MOVE.HEAD.FAST lookup_startpos 0
    CALL G_WAITZERO
    WAIT:500

    if (Pink_Ball>0) then
      CALL GSD_TRACKBALL
      CALL GSD_GETBALLPOS
    endif

    if (Pink_Ball==0) then
      if (F_ABS(lookup_startpos)>50) then
        PLAY ACTION MOVE.HEAD.SLOW 0 0
      else 
        PLAY ACTION MOVE.HEAD.SLOW 90*target_side 0
      endif

      if (FSD_WAITSCAN()==GO_SEEBALL) then
        SET lookup_startpos Head_Pan
        CALL GSD_GETBALLPOS
      endif
    endif

    // Are we in position?  If so, turn to face ball straight on...
    if (Pink_Ball>0) && (ball_dist>0) then
      if (target_side) ? (F_ABS(ball_horz)<70) : (F_ABS(ball_horz)<30) then
        CALL GSD_TRACKBALL
        while (Pink_Ball>0) && (Wait>0) && (F_ABS(Head_Pan)>10)
	  SET init_wait Wait
          PLAY ACTION TURN 10*F_SIGN(Head_Pan)
          CALL G_WAIT
          WAIT:100
        wend
        GO SD_ALIGNBALL
      endif
    endif

    // Look back at disc (turning off ball tracking), advance & turn...
    PLAY ACTION MOVE.HEAD.FAST disc_pan disc_tilt
    CALL G_WAITZERO

    for i 1 3
      PLAY ACTION WALK -20*target_side 20
      CALL G_WAITZERO
    next

    PLAY ACTION TURN 20*target_side 
    CALL G_WAITZERO

    SWITCH:FSD_TRACKHEAD(COLORMASK_BLUE,align_x,align_y)
      CASE:GO_IDLE GO SD_FINDBALL
      CASE:GO_SEEBALL GO SD_TOWERLOCK
    GO SD_ALIGNLOOP
  endif


//
// Step 6.  We are now looking at disc, and we're at the correct end 
// of the charger.  
//
// Now to align AIBO's body straight-on with the charger.  The ideal
// thing would be turning on the Z-axis of the nose-camera (when
// pointed straight down).  Alas, the turn Z-axis is the body center.  
// Therefore we walk back & forth at slight offset angles until 
// alignment is achieved.
//
// Some custom actions of camera Z-axis turns would definately simplify 
// things, and make this procedure seem less mechanical.  It'd also
// provide some immunity against the walking styles changing again.
//
:SD_ALIGNBALL
  CALL GSD_FOCUS_BALL

  for i 1 3 
    SWITCH i
      CASE:1 PLAY ACTION MOVE.HEAD.FAST 0 0
      CASE:2 PLAY ACTION MOVE.HEAD.FAST 35 0
      CASE:3 PLAY ACTION MOVE.HEAD.FAST -35 0
    if (FSD_WAITSCAN()==GO_SEEBALL) SD_ALIGNSEEBALL
  next

:SD_ALIGNRETRY
  PLAY ACTION WALK 180 80			// Can't see ball... backup & try again...
  PLAY ACTION MOVE.HEAD.FAST disc_pan 0
  CALL G_WAIT
  GO SD_FINDBALL


// 
// The disc is at our feet, and the tower alignment ball is visible 
// within a narrow arc of our vision...
//
:SD_ALIGNSEEBALL
  WAIT 500					// In case a false ball detect got us here,
  if (Pink_Ball==0) SD_ALIGNBALL		// double check...

  CALL GSD_GETBALLPOS
  if (F_ABS(ball_horz)>45) SD_ALIGNRETRY

  // Turn until looking straight at ball.  If unable to line up
  // after 20 steps, retry the alignment...
  CALL GSD_TRACKBALL
  if (F_ABS(Head_Pan+Pink_Ball_Horz)>5) then
    if (!FSD_SLOWTURN2BALL(20)) SD_ALIGNRETRY
  endif

  align_x := 44
  align_count := 0
  align_hint := 0

  // See where disc is.  Search for it to reaquire if not visible...
  CALL GSD_FOCUS_BLUE

  // If already passed through disc alignment once, and didn't move to align with ball,
  // then we are ready to approach the charger...
  if (aligned_disc) SD_APPROACH

  // Look for the disc...
  for i 1 4
    SWITCH i
      CASE:1 PLAY ACTION MOVE.HEAD.FAST 0 -90 
      CASE:2 PLAY ACTION MOVE.HEAD.FAST 20 -60 
      CASE:3 PLAY ACTION MOVE.HEAD.FAST 0 -60 
      CASE:4 PLAY ACTION MOVE.HEAD.FAST -20 -60 
    CALL G_WAITZERO

    SWITCH:FSD_TRACKHEAD(COLORMASK_BLUE,align_x,36)
      CASE:GO_AHEAD GO SD_ALIGNSIDESTEP
      CASE:GO_SEEBALL GO SD_TOWERLOCK
  next

  if (FSD_REAQUIRE_DISC()==GO_IDLE) SD_FINDBALL


:SD_TRACKSIDESTEP
  SWITCH:FSD_TRACKHEAD(COLORMASK_BLUE,align_x,36)
    CASE:GO_AHEAD GO SD_ALIGNSIDESTEP
    CASE:GO_SEEBALL GO SD_TOWERLOCK

  SWITCH:align_hint
    CASE:0 if (FSD_REAQUIRE_DISC()==GO_IDLE) SD_FINDBALL
    CASE:1 if (FSD_REAQUIRE_LEFT()==GO_IDLE) SD_FINDBALL
    CASE:2 if (FSD_REAQUIRE_RIGHT()==GO_IDLE) SD_FINDBALL
    CASE:3 if (FSD_REAQUIRE_FORWARD()==GO_IDLE) SD_FINDBALL
    CASE:4 if (FSD_REAQUIRE_BACKWARD()==GO_IDLE) SD_FINDBALL

  SWITCH:FSD_TRACKHEAD(COLORMASK_BLUE,align_x,36)
    CASE:GO_IDLE GO SD_FINDBALL
    CASE:GO_SEEBALL GO SD_TOWERLOCK

:SD_ALIGNSIDESTEP
  step_count := 0

  // If battery gets too low, then shutdown now...
  if (Back_ON>0) || F_NEED2REST() then
    PLAY ACTION MOVE.HEAD.FAST Head_Pan Head_Tilt	// turn off tracking
    CALL G_WAITZERO
    RET:GO_RESTPOS
  endif

  // Get up close to disc...
  if (disc_tilt>-60) then
    aligned_disc := FALSE // can't be aligned if can't see disc
    while (disc_tilt>-70) // go little further than necessary for hysteresis
      PLAY ACTION WALK 0 20
      CALL G_WAITZERO
      align_hint := 4
      ++align_count
      ++step_count

      // update disc_tilt...
      SWITCH:FSD_TRACKHEAD(COLORMASK_BLUE,align_x,36)
        CASE:GO_AHEAD GO SD_ALIGNSIDESTEP
        CASE:GO_SEEBALL GO SD_TOWERLOCK

      if (FSD_REAQUIRE_BACKWARD()==GO_IDLE) SD_FINDBALL
    wend
    GO SD_TRACKSIDESTEP
  endif

  // Backoff if too close to disc...
  if (disc_tilt<-90) then
    while (disc_tilt<-90)
      PLAY ACTION WALK 180 20
      CALL G_WAITZERO
      align_hint := 3
      ++align_count
      ++step_count

      // update disc_tilt...
      SWITCH:FSD_TRACKHEAD(COLORMASK_BLUE,align_x,36)
        CASE:GO_AHEAD GO SD_ALIGNSIDESTEP
        CASE:GO_SEEBALL GO SD_TOWERLOCK

      if (FSD_REAQUIRE_FORWARD()==GO_IDLE) SD_FINDBALL
    wend
    GO SD_TRACKSIDESTEP
  endif

  // Reverify alignment with ball if took steps (sometimes throws off AIBO), or made
  // several side-step adjustments...
  if (step_count>0) || (align_count>2) SD_ALIGNBALL

  // Move sideways to realign with disc...
  SWITCH 0
  if (Posture1 == POSTURE_WALK) then
    if (LFLeg_Lat<8) && (RFLeg_Lat>14) then // leaning to left
      PRINT "compensating for walk pose lean to left."
      disc_pan += 3
    endif
    if (RFLeg_Lat<8) && (LFLeg_Lat>14) then // leaning to right
      PRINT "compensating for walk pose lean to right."
      disc_pan -= 3
    endif
    CSET:disc_pan >= 17:1
    CSET:disc_pan >= 13:2
    CSET:disc_pan >= 7:3
    CSET:disc_pan <= -17:4
    CSET:disc_pan <= -13:5
    CSET:disc_pan <= -7:6
  else
    CSET:disc_pan >= 24:1
    CSET:disc_pan >= 20:2
    CSET:disc_pan >= 14:3
    CSET:disc_pan <= -12:4
    CSET:disc_pan <= -8:5
    CSET:disc_pan <= -2:6
  endif

  if (Context>0) then
    align_hint := (Context>3) ? 1 : 2
    PRINT "SideStep  step_index=%d  disc_pan=%d  Head_Pan=%d" Context disc_pan Head_Pan
    CASE:1:PLAY ACTION step_left
    CASE:2:PLAY ACTION step_left1
    CASE:3:PLAY ACTION step_left2
    CASE:4:PLAY ACTION step_right
    CASE:5:PLAY ACTION step_right1
    CASE:6:PLAY ACTION step_right2
    CALL G_WAITZERO
    ++align_count
    GO SD_TRACKSIDESTEP
  endif

  // Only proceed if there were no adjustments above.  If there were,
  // reconfirm alignment with tower ball...
  if (align_count>0) then
    aligned_disc := TRUE
    GO SD_ALIGNBALL
  endif



//
// Step 7.  Should now be standing straight-on aligned with charger.
// Walk forward until distance is 300mm (or less) from ball on alignment tower.
//
// Next, do single steps 4 times (recording minimum head pan), and continue 
// single steps until find minimum head pan again.  This fine-tunes our
// alignment before attempting to dock with charger.
//
:SD_APPROACH
  SET init_wait Wait
  PLAY ACTION happy_eyes
  PLAY ACTION MOVE.HEAD.FAST 0 0
  CALL GSD_FOCUS_BALL
  CALL G_WAIT

  CALL GSD_TRACKBALL

  for i 1 4
    sd_maxstep := 15
    sd_turnstep := 0
    while (Pink_Ball>0) && (FSD_GETBALLDIST()>300) && (sd_maxstep>0) && (Wait>0)
      --sd_maxstep
      if (--sd_turnstep<0) then
        sd_turnstep := 2
      endif

      // If get too far out of whack, back off and try again...
      CALL GSD_GETBALLPOS
      if (F_ABS(ball_horz)>=20) then
        CALL GSD_BACKOFF
        GO SD_FINDBALL
      endif

      // Turn to ball.  If unable to lineup after four turn steps, backup...
      if (!FSD_SLOWTURN2BALL(4)) then
        CALL GSD_BACKOFF
        GO SD_FINDBALL
      endif

      SET init_wait Wait
      PLAY ACTION WALK 0 20
      CALL G_WAIT
      WAIT 1500
    wend

    // Lock back onto alignment tower ball if walk caused us to loose track...
    CALL GSD_TRACKBALL
  next

  // If lost alignment tower ball, go back & look for alignment disc...
  if (Pink_Ball==0) || (Wait==0) SD_FINDBALL

  PLAY ACTION mission_impossible
  SET init_wait Wait

  // Do couple of single steps & establish minimum possible aligned Head_Pan...
  SET min_pan 999
  for i 1 2
    PLAY ACTION WALK 0 20
    CALL G_WAIT
    WAIT 500 // settling time
    pan := F_ABS(Head_Pan)
    if (pan<min_pan) then
      min_pan := pan
    endif
  next
 
  // Continue to do single steps until achieve minumum possible Head_Pan alignment...
  SET init_wait Wait
  while (pan>(min_pan+5))
    i := 4
    SET min_pan2 999
    while (pan>(min_pan+5)) && (i>0)
      PLAY ACTION WALK 0 20
      CALL G_WAIT
      WAIT 500 // settling time
      pan := F_ABS(Head_Pan)
      if (pan<min_pan2) then
        min_pan2 := pan
      endif
      SUB i 1
    wend
    min_pan := min_pan2
  wend



//
// Step 8.  Attempt docking.  Might fail we arn't aligned.  If it
// doesn't work, stand up, move forward a bit, and try again.
// If cannot dock after five tries, give up (back off the charger).
// AIBO will almost always succeed on the first attempt.
//
:SD_DOCK
  VSAVE dockdir 	// remember search direction for next time
  WAIT 1000
  CALL GSD_TRACKBALL

  // Lost the ball?  Go find it...
  if (Wait==0) || (Pink_Ball==0) SD_FINDBALL

  // If get too wildly out of whack, we must have missed the charger.
  // Backup and try again...
  CALL GSD_GETBALLPOS
  if (F_ABS(ball_horz)>15) then
    CALL GSD_BACKOFF
    GO SD_FINDBALL
  endif

  SET init_wait Wait
  PLAY ACTION charger_mount
  CALL G_WAIT

  // Give AIBO time to realize its on the charger...
  WAIT 4000

  // If not in charger posture & didn't do annoying panic shutdown, 
  // then we didn't succeed.  Stand up, move a bit, and try again...
  if ((Posture1 != POSTURE_CHARGE1) && (Posture1 != POSTURE_CHARGE2)) then
    ++sd_dockcount	
    PLAY ACTION charger_dismount
    CALL G_WAIT

    // Lost the ball?  Go find it...
    if (Wait==0) || (Pink_Ball==0) then 
      GO SD_FINDBALL
    endif

    // More than 6 attempts to dock fail, give it up.  Lie down near
    // charger and shutdown...
    if (sd_dockcount>=6) then
      PLAY ACTION WALK 180 500
      CALL G_WAIT
      PLAY ACTION LIE
      CALL G_WAIT
      GO SD_SHUTDOWN
    endif

    // Try walking an extra step forward.  After third attempt backoff
    // and reapproach the charger.  Might have missed charger, or somehow
    // walked too far onto it.
    if (sd_dockcount != 4) then
      PLAY ACTION WALK 0 20
      CALL G_WAIT
      GO SD_DOCK
    else
      PLAY ACTION WALK 180 200
      CALL G_WAIT
      GO SD_APPROACH
    endif
  endif



//
// Step 9.  Shut ourself off (assuming Sony doesn't do the panic shutdown first).
//
// Ideally we would like to set the snooze alarm for an hour or two, wake up all
// rested & refreshed, back off the charger, and continue in our doggish ways.  
//
// Alas, not currently possible because in RCode 1.2 Sony BROKE the charger mode.  
// AIBO originally remained awake on the charger,  It's something that USED TO WORK...
// 
// :-(
//
:SD_SHUTDOWN
  // Put head down...
  RND pan -25 25
  PLAY ACTION MOVE.HEAD.FAST pan -90
  CALL G_WAIT
  WAIT 1000

  // Nighty night... ZZZzzzz...
  HALT



//////////////////////////////////////////////////////////////////////////////////
//
// Stall until Wait goes to zero, 30 seconds elapse, or see ball (in which case
// stop the head moving)...
//
:FSD_WAITSCAN
  SET Clock 0
  while (Wait>0) && (Clock<1024) && (Pink_Ball==0)
    WAIT 1
  wend

  // Stop head scan...
  PLAY ACTION MOVE.HEAD.FAST Head_Pan Head_Tilt
  CALL G_WAITZERO

  // Start tracking...
  CALL GSD_TRACKBALL
  RETURN:(Pink_Ball>0)?GO_SEEBALL:GO_IDLE



//////////////////////////////////////////////////////////////////////////////////
//
// Wait while head track locks onto ball (maximum of 3 seconds)...
//
:GSD_TRACKBALL
  LOCAL trackcount 0
  while (Pink_Ball>0) && (Wait==0) && (++trackcount<4)
    PLAY ACTION TRACK_HEAD PINK_BALL
    CALL GSD_WAITTRACK
  wend
  RET:1

:GSD_WAITTRACK
  LOCAL old_pan Head_Pan
  LOCAL old_tilt Head_Tilt
  LOCAL count 0

  if (Pink_Ball>0) && (Wait>0) then
    WAIT 1000
  endif

  // While head thrashing around, wait here...
  while (Pink_Ball>0) && (Wait>0) && ((F_ABS(Head_Pan-old_pan)>5) || (F_ABS(Head_Tilt-old_tilt)>5)) && (++count<=2)
    old_pan := Head_Pan
    old_tilt := Head_Tilt
    WAIT 1000
  wend
  RET:1


//////////////////////////////////////////////////////////////////////////////////
//
// Stall until Wait goes to zero, 30 seconds elapse, or see obstacle 
// (in which case stop walking)...
//
:GSD_WAITWALK
  SET Clock 0
  while (Wait>0) && (Clock<1024) && (Distance>TOOCLOSEDIST)
    WAIT 1
  wend
  if (Distance<=TOOCLOSEDIST) then 
    PLAY ACTION WALK 0 0
    CALL G_WAITZERO
    RET:GO_STOP
  endif
  RET:GO_WALK


//////////////////////////////////////////////////////////////////////////////////
//
// Adjust position if not in walking posture, to compensate for the initial
// off-center lunge to the left that AIBO makes on first step.
//
:GSD_INITWALK
  if (Posture1 == POSTURE_STAND) then // compensate for walk posture offseting AIBO to the left
    aligned_disc := FALSE // if move, then no longer aligned to disc
    SET init_wait Wait
    PLAY ACTION step_right2
    CALL G_WAIT
  endif
  RET:1


//////////////////////////////////////////////////////////////////////////////////
//
// Back away from the ball...
//
:GSD_BACKOFF
  LOCAL maxstep 20
  while (Pink_Ball>0) && (FSD_GETBALLDIST()<650) && (maxstep>0)
    --maxstep
    SET init_wait Wait
    PLAY ACTION WALK 180 60
    CALL G_WAIT
    WAIT 1500 // settling time
  wend
  PLAY ACTION MOVE.HEAD.FAST 0 0 // stop tracking ball
  CALL GSD_FOCUS_BLUE
  CALL G_WAITZERO
  RET:1



//////////////////////////////////////////////////////////////////////////////////
//
// Get ball position in our field of view...
//
:GSD_GETBALLPOS
  ball_dist := FSD_GETBALLDIST()
  ball_horz := Head_Pan + Pink_Ball_Horz
  ball_vert := Head_Tilt + Pink_Ball_Vert
  if (LFLeg_Lat<8) && (RFLeg_Lat>14) && (ball_dist<400) then
    ball_horz -= 3
  endif
  if (RFLeg_Lat<8) && (LFLeg_Lat>14) && (ball_dist<400) then
    ball_horz += 3
  endif
  PRINT "ball: dist=%d horz=%d vert=%d",ball_dist,ball_horz,ball_vert
  RET:1



//////////////////////////////////////////////////////////////////////////////////
//
//  Set new camera settings...
//
:FSD_SETDEVCTL
  ARG:new_balance
  ARG:new_gain
  ARG:new_speed
  if (new_balance>0) && (new_gain>0) && (new_speed>0) then
    if (new_balance!=old_balance) || (new_gain!=old_gain) || (new_speed!=old_speed) then
      if (new_balance!=old_balance) then
        AP_DEVCTL 1 new_balance
        old_balance := new_balance
      endif
      if (new_gain!=old_gain) then
        AP_DEVCTL 2 new_gain
        old_gain := new_gain
      endif
      if (new_speed!=old_speed) then
        AP_DEVCTL 3 new_speed
        old_speed := new_speed
      endif
      WAIT:FOCUS_SPEED
    endif
  endif
  RETURN:1



//////////////////////////////////////////////////////////////////////////////////
//
//  Select default camera settings (for detecting ball)...  This is not
//  autodetected, since if Aibo cannot normally see the ball in the 
//  current lighting conditions its either way too bright or dark.
//
:GSD_FOCUS_BALL
  FSD_SETDEVCTL(3,2,2)   // A guess...  What is AIBO's normal default settings?
  RET:1



//////////////////////////////////////////////////////////////////////////////////
//
//  Select camera settings most likely to see blue (favorite color)...
//
:GSD_FOCUS_BLUE
  FSD_SETDEVCTL(max_blue_balance, max_blue_gain, max_blue_speed)
  RET:1



//////////////////////////////////////////////////////////////////////////////////
//
//  Select camera settings most likely to see blue (favorite color)...
//
:GSD_FOCUS_GREEN
  FSD_SETDEVCTL(max_green_balance, max_green_gain, max_green_speed)
  RET:1



//////////////////////////////////////////////////////////////////////////////////
//
// Find best settings for tracking pink, blue & green colors.
// Once found, return x,y coordinates of centroid for each color.
//
:GSD_COLORCHECK
  LOCAL i
  LOCAL j
  LOCAL k

  SET old_balance -999
  SET old_gain -999
  SET old_speed -999

  // If user has specified manual settings for the camera, use those instead of self-calibrating)...
  if !sd_colorscan && behavior[BEHAVIOR_CAMERA_MODE] then
    sd_colorscan := TRUE

    max_blue_level := BLUE_THRESHOLD
    max_blue_balance := behavior[BEHAVIOR_CAMERA_BALANCE]
    max_blue_gain := behavior[BEHAVIOR_CAMERA_GAIN]
    max_blue_speed := behavior[BEHAVIOR_CAMERA_SHUTTER]

    max_green_level := GREEN_THRESHOLD
    max_green_balance := behavior[BEHAVIOR_CAMERA_BALANCE]
    max_green_gain := behavior[BEHAVIOR_CAMERA_GAIN]
    max_green_speed := behavior[BEHAVIOR_CAMERA_SHUTTER]

    SET Eye_L3 0
    SET Eye_L2 0
    SET Eye_L1 0
    SET Eye_R1 0
    SET Eye_R2 0
    SET Eye_R3 0
    WAIT:500
  endif

  // If need camera calibration, proceed now...
  if (!sd_colorscan) then
    SET sd_colorscan TRUE

    SET max_blue_level 0
    SET max_blue_balance 0
    SET max_blue_gain 0
    SET max_blue_speed 0

    SET max_green_level 0
    SET max_green_balance 0
    SET max_green_gain 0
    SET max_green_speed 0

    SET blue_level 0
    SET green_level 0

    SET spinner 0
    SET trackcount 0

    // Get whatever settings ball is returning using default camera settings...
    AP_GETCOLORARRAY colorarray
    AP_COLORFND colorarray COLORMASK_PINK pink_x pink_y pink_level

    for i 1 3		// white balance
      AP_DEVCTL 1 i
      WAIT:FOCUS_SPEED	// Extra stall when changing white balance...

      for k 1 3		// shutter speed
        AP_DEVCTL 3 k

        for j 1 3	// camera gain
          AP_DEVCTL 2 j

	  // Do LED spinner on face to let owner know AIBO is thinking...
	  #ifdef RCODE250
	  SWITCH (spinner++)
	    CASE 0 : SET Eye_R3 0
	    CASE 0 : SET Eye_L3 1
	    CASE 1 : SET Eye_L3 0
	    CASE 1 : SET Eye_L2 1
	    CASE 2 : SET Eye_L2 0
	    CASE 2 : SET Eye_L1 1
	    CASE 3 : SET Eye_L1 0
	    CASE 3 : SET Eye_R1 1
	    CASE 4 : SET Eye_R1 0
	    CASE 4 : SET Eye_R2 1
	    CASE 5 : SET Eye_R2 0
	    CASE 5 : SET Eye_R3 1
	    CASE 5 : SET spinner 0
	  #endif

  	  WAIT:FOCUS_SPEED // wait for camera to settle 
          AP_GETCOLORARRAY colorarray
          AP_COLORFND colorarray COLORMASK_GREEN green_x green_y green_level
          AP_COLORFND colorarray COLORMASK_BLUE blue_x blue_y blue_level

	  #define TRACKINDEX(i,j,k) ((k-1)*54 + (j-1)*18 + (i-1)*6)
          #define BLUE_LEVEL 0
          #define BLUE_X 1
          #define BLUE_Y 2
          #define GREEN_LEVEL 3
          #define GREEN_X 4
          #define GREEN_Y 5

	  trackindex := TRACKINDEX(i,j,k)
          trackarray[trackindex+BLUE_LEVEL] := blue_level
          trackarray[trackindex+BLUE_X] := blue_x
          trackarray[trackindex+BLUE_Y] := blue_y
          trackarray[trackindex+GREEN_LEVEL] := green_level
          trackarray[trackindex+GREEN_X] := green_x
          trackarray[trackindex+GREEN_Y] := green_y

	  PRINT "--- camera calibration %d %d %d --- " i j k
	  PRINT "ball=%d bright=%d p,b,g=%d %d %d" Pink_Ball Brightness pink_level blue_level green_level

	  // Can't be too intense, and centroid of blue must be below pink...
          if (blue_level>max_blue_level) && (blue_level<MAX_THRESHOLD) then
            //if (F_ABS(blue_x-pink_x)<=16) && (pink_y<blue_y) then
            if (F_ABS(blue_x-pink_x)<=16) && (pink_y<blue_y) then
              SET max_blue_level blue_level
    	      SET max_blue_balance i
  	      SET max_blue_gain j
	      SET max_blue_speed k
            endif
          endif

	  // Can't be too intense, and centroid of green must be below pink...
          if (green_level>max_green_level) && (green_level<MAX_THRESHOLD) then
            if (F_ABS(green_x-pink_x)<=16) && (pink_y<green_y) && ((blue_y<0) || ((pink_y<blue_y) && (blue_y<green_y))) then
              SET max_green_level green_level
    	      SET max_green_balance i
	      SET max_green_gain j
	      SET max_green_speed k
            endif
          endif
        next
      next
    next

    // Turn off LED's...
    SET Eye_L3 0
    SET Eye_L2 0
    SET Eye_L1 0
    SET Eye_R1 0
    SET Eye_R2 0
    SET Eye_R3 0

    // If blue sampled above threshold consistently, then choose lowest value...
    if (max_blue_level==0) then
      PRINT "Reevaluating for minimum blue level setting"
      max_blue_level := 999
      for k 1 3
        for j 1 3
          for i 1 3
	    trackindex := TRACKINDEX(i,j,k)
	    blue_level := trackarray[trackindex+BLUE_LEVEL]
	    blue_x := trackarray[trackindex+BLUE_X]
	    blue_y := trackarray[trackindex+BLUE_Y]
	    if (blue_level>0) && (blue_level<max_blue_level) then
              if (F_ABS(blue_x-pink_x)<=16) && (pink_y<blue_y) then
	        SET max_blue_level blue_level
  	        SET max_blue_balance i
	        SET max_blue_gain j
	        SET max_blue_speed k
	      endif
	    endif
          next
        next
      next
    endif

    // If blue is way stronger than pink (more than 3X), then look for camera settings 
    // with a little less gain (to avoid background color/saturation issues)...
    if (max_blue_level > 3*pink_level) then
      PRINT "Reevaluating for lower blue level setting"
      new_blue_level := BLUE_THRESHOLD
      for k 1 3
        for j 1 3
          for i 1 3
	    trackindex := TRACKINDEX(i,j,k)
	    blue_level := trackarray[trackindex+BLUE_LEVEL]
	    blue_x := trackarray[trackindex+BLUE_X]
	    blue_y := trackarray[trackindex+BLUE_Y]
	    if (blue_level>new_blue_level) && (blue_level<=3*pink_level) then
              if (F_ABS(blue_x-pink_x)<=16) && (pink_y<blue_y) then
                SET new_blue_level blue_level
	        SET max_blue_level blue_level
  	        SET max_blue_balance i
	        SET max_blue_gain j
	        SET max_blue_speed k
	      endif
	    endif
          next
        next
      next
    endif

    // If green sampled above threshold consistently, then choose lowest value...
    if (max_green_level==0) then
      PRINT "Reevaluating for minimum green level setting"
      max_green_level := 999
      for k 1 3
        for j 1 3
          for i 1 3
	    trackindex := TRACKINDEX(i,j,k)
	    green_level := trackarray[trackindex+GREEN_LEVEL]
	    green_x := trackarray[trackindex+GREEN_X]
	    green_y := trackarray[trackindex+GREEN_Y]
	    if (green_level>0) && (green_level<max_green_level) then
              if (F_ABS(green_x-pink_x)<=16) && (pink_y<green_y) then
	        SET max_green_level green_level
  	        SET max_green_balance i
	        SET max_green_gain j
	        SET max_green_speed k
	      endif
	    endif
          next
        next
      next
    endif

    // If green is way stronger than pink (more than 3X), then look for camera settings 
    // with a little less gain (to avoid background color/saturation issues)...
    if (max_green_level > 3*pink_level) then
      PRINT "Reevaluating for lower green level setting"
      new_green_level := GREEN_THRESHOLD
      for k 1 3
        for j 1 3
          for i 1 3
	    trackindex := TRACKINDEX(i,j,k)
	    green_level := trackarray[trackindex+GREEN_LEVEL]
	    green_x := trackarray[trackindex+GREEN_X]
	    green_y := trackarray[trackindex+GREEN_Y]
	    if (green_level>new_green_level) && (green_level<=3*pink_level) then
              if (F_ABS(green_x-pink_x)<=16) && (pink_y<green_y) then
                SET new_green_level green_level
  	        SET max_green_level green_level
  	        SET max_green_balance i
	        SET max_green_gain j
	        SET max_green_speed k
	      endif
	    endif
          next
        next
      next
    endif
  endif

:GSD_LEVELCHECK
  SET pink_x -1
  SET pink_y -1
  SET blue_x -1
  SET blue_y -1
  SET green_x -1
  SET green_y -1
  SET see_pink_ball 0

  CALL GSD_BALL_LEVEL
  if (max_green_level>0) CALL GSD_GREEN_LEVEL
  if (max_blue_level>0) CALL GSD_BLUE_LEVEL

  if (pink_x<0) then
    pink_level := 0
  endif

  if (green_x<0) then
    green_level := 0
  endif

  if (blue_x<0) then
    blue_level := 0
  endif

  PRINT "---"
  PRINT "pink:  l=%d  x,y=%d,%d" pink_level pink_x pink_y
  PRINT "green: l=%d  b=%d  s=%d  x,y=%d,%d" green_level max_green_balance max_green_speed green_x green_y
  PRINT "blue:  l=%d  b=%d  s=%d  x,y=%d,%d" blue_level max_blue_balance max_blue_speed blue_x blue_y
  RET:1


:GSD_BALL_LEVEL
  CALL GSD_FOCUS_BALL
  AP_GETCOLORARRAY colorarray
  AP_COLORFND colorarray COLORMASK_PINK pink_x pink_y pink_level
  SET pink_level Pink_Ball*PINK_THRESHOLD
  RET:1

:GSD_GREEN_LEVEL
  CALL GSD_FOCUS_GREEN
  AP_GETCOLORARRAY colorarray
  AP_COLORFND colorarray COLORMASK_GREEN green_x green_y green_level
  RET:1

:GSD_BLUE_LEVEL
  CALL GSD_FOCUS_BLUE
  AP_GETCOLORARRAY colorarray
  AP_COLORFND colorarray COLORMASK_BLUE blue_x blue_y blue_level
  RET:1



//////////////////////////////////////////////////////////////////////////////////
//
//  Function returns TRUE if high probability we are seeing alignment disc.
//  If found, camera is configured to focus on blue (fav color) used to
//  position head.  Check twice...
//
:FSD_SEE_DISC
  LOCAL count 0
  while (count<2)
    SET pink_level 0
    SET green_level 0
    SET blue_level 0

    if (Head_Tilt>-10) then
      CALL GSD_BALL_LEVEL
      if (Pink_Ball>0) then
        return FALSE
      endif
    endif

    CALL GSD_GREEN_LEVEL
    if (green_level<GREEN_THRESHOLD) then
      return FALSE
    endif

    CALL GSD_BLUE_LEVEL
    if (blue_level<BLUE_THRESHOLD) then
      return FALSE
    endif

    ++count
  wend

  PRINT "---"
  PRINT "pink:  l=%d  x,y=%d,%d" pink_level pink_x pink_y
  PRINT "green: l=%d  b=%d  s=%d  x,y=%d,%d" green_level max_green_balance max_green_speed green_x green_y
  PRINT "blue:  l=%d  b=%d  s=%d  x,y=%d,%d" blue_level max_blue_balance max_blue_speed blue_x blue_y
  return TRUE



//////////////////////////////////////////////////////////////////////////////////
//
//  Function returns TRUE if need to turn towards ball...
//
:FSD_MUST_TURN
  ball_horz := Head_Pan // + Pink_Ball_Horz
  return (ball_horz<-15) || (ball_horz>15)



//////////////////////////////////////////////////////////////////////////////////
//
//  Function returns TRUE if need to move to/from the ball...  Averages
//  the pink ball dist to smooth out jitter...
//
:FSD_MUST_MOVE
  ball_dist := FSD_GETBALLDIST()
  return (ball_dist<MIN_BALL_DIST) || (ball_dist>MAX_BALL_DIST)


:FSD_GETBALLDIST
  LOCAL bdist
  WAIT:50
  SET bdist Pink_Ball_Dist
  WAIT:50
  ADD bdist Pink_Ball_Dist
  DIV bdist 2
  return bdist



//////////////////////////////////////////////////////////////////////////////////
//
// Turn AIBO until looking straight at ball
//
:GSD_ALIGN2BALL
  CALL GSD_TRACKBALL
  while (FSD_MUST_TURN()) 
    if (Pink_Ball==0) then // lost ball
      PLAY ACTION MOVE.HEAD.FAST Head_Pan Head_Tilt // turn off tracking
      CALL G_WAITZERO
      RET:1
    endif
    CALL GSD_TRACKBALL
    CALL GSD_TURN2BALL
  wend

  PLAY ACTION MOVE.HEAD.FAST Head_Pan Head_Tilt	// turn off tracking
  GO G_WAITZERO



//////////////////////////////////////////////////////////////////////////////////
//
//  Turn so pointing at the ball...
//
:GSD_TURN2BALL
  // Make sure tracking is online...
  CALL GSD_TRACKBALL
  if (Pink_Ball==0) then // lost view of ball
    RET:GO_IDLE
  endif

  if (!FSD_MUST_TURN()) then // already close enough
    RET:GO_SEEBALL
  endif

  turn_angle := ball_horz/2
  if (ball_horz>0) && (turn_angle<10) then
    turn_angle := 10
  endif
  if (ball_horz<0) && (turn_angle>-10) then
    turn_angle := -10
  endif

  SET init_wait Wait
  PLAY ACTION TURN turn_angle

  while (Wait>0)
    WAIT:1
    if (Pink_Ball==0) then // lost view of ball
      CALL G_WAIT
      RET:GO_IDLE
    endif

    // Turn completed?
    if (Wait<=init_wait) then 
      RET:GO_SEEBALL
    endif
  wend

  RET (Pink_Ball)?GO_SEEBALL:GO_IDLE



//////////////////////////////////////////////////////////////////////////////////
//
//  Function to turn slowly, so we are pointing as accurately at ball as possible.
//  Returns TRUE if aligned.  FALSE if took too many steps (or lost ball)...
//
:FSD_SLOWTURN2BALL
  ARG maxsteps
  LOCAL i 0
  LOCAL pan 0
  LOCAL turnhistory 0
  LOCAL stepcount 0

  WAIT 500 // settling time
  if (F_ABS(Head_Pan)<=5) then
    return TRUE
  endif

  for i 1 3
    // If lost ball, abort...
    if (Pink_Ball==0) then
      return FALSE
    endif

    // If taking too long, quit.  However, if all we were doing is turning
    // back & forth, then its not a problem (just wasting time)...
    if (stepcount>=maxsteps) then
      return (turnhistory==0x5) || (turnhistory==0xA)
    endif

    // Make sure tracking is online...
    CALL GSD_TRACKBALL

    while (stepcount<maxsteps) && (Pink_Ball>0) && (Wait>0) && (F_ABS(Head_Pan)>6)
      // If we're just turning left/right back & forth, then quit now...
      if (stepcount>=4) && ((turnhistory==0x5) || (turnhistory==0xA)) then
        return TRUE
      endif

      SET init_wait Wait
      aligned_disc := FALSE // if move, then no longer aligned to disc
      if (Head_Pan>0) then
        turnhistory := ((turnhistory*2)&0xF)+1
	PLAY ACTION TURN 10
      else
        turnhistory := ((turnhistory*2)&0xF)
	PLAY ACTION TURN -10
      endif
      CALL G_WAIT
      WAIT 1000 // settling time
      ++stepcount
    wend
  next
  return TRUE



//////////////////////////////////////////////////////////////////////////////////
//
//  Move so we are 700mm from ball.  If too close, back away.
//
:GSD_MOVE2BALL
  // Make sure tracking is online...
  CALL GSD_TRACKBALL

  ball_dist := (FSD_GETBALLDIST()+FSD_GETBALLDIST())/2

  // If have performed color scan already, go for ideal distance (places AIBO
  // distance from ball equal to length of charging station).  If havent done
  // color scan yet, then get a little closer...
  if (sd_colorscan) then
    walk_dist := (ball_dist-IDEAL_BALL_DIST)/2
  else
    walk_dist := (ball_dist-MIN_BALL_DIST)/3
  endif

  PRINT "move2ball ball_dist=%d  walk_dist=%d" ball_dist walk_dist

  WAIT 1
  SET init_wait Wait

  if (walk_dist<0) then
    if (walk_dist>-20) then
      walk_dist := -20
    endif
    if (walk_dist<-100) then
      walk_dist := -100
    endif
    PLAY ACTION WALK 180 -walk_dist
  else 
    if (walk_dist<20) then
      walk_dist := 20
    endif
    if (walk_dist>100) then
      walk_dist := 100
    endif
    PLAY ACTION WALK 0 walk_dist
  endif

  while (Wait>init_wait)
    WAIT:1
    if (Pink_Ball==0) || (F_ABS(Head_Pan)>50) then
      PLAY ACTION WALK 0 0
      CALL G_WAIT
      RET (Pink_Ball)?GO_SEEBALL:GO_IDLE
    endif
  wend

  RET (Pink_Ball)?GO_SEEBALL:GO_IDLE



//////////////////////////////////////////////////////////////////////////////////
//
//  Attempt to center head on the target color (typically the blue/green alignment 
//  disc).  Scan for the colors, then center head on the color centroid.
//
//  Target x,y centering coordinates are provided as parameters.
//
//  This head centering business is slightly complex, since AIBO's head movements
//  tends to be "sticky" sometimes (jerks from stationary instead of moving smoothly).
//  It leads to cycling where the head snaps from off-left to off-right & visa-versa 
//  (for example), and never truely locks-on like its supposed to.  Ug...  
// 
//  Sticking is detected by looking for cases where movement doesn't occur (when expected).
//  Cycling typically has two end points.  Groups of near identical coordinates are tracked.
//  If there are multiple hits to group A, then group B, then group A again, sticking is
//  assumed and tracking is stopped.
//
//  Function Return:
//    GO_IDLE : Lost lock on the disc
//    GO_AHEAD : Disc centered (or as close as possibly centered)
//
// FUTURE:
//   A lookup table taking centroid & current head position as inputs, and returning
//   best guess head movements (to permit faster centering).
//
:FSD_TRACKHEAD
  ARG trackmask
  ARG xtarget
  ARG ytarget
  LOCAL minor_count 0
  LOCAL last_headpan -999
  LOCAL last_pan2 -999
  LOCAL xdelta -999
  LOCAL ydelta -999
  LOCAL xadjust -999
  LOCAL yadjust -999
  SET pan Head_Pan
  SET tilt Head_Tilt

  SET track_x -999
  SET track_y -999
  SET track_level 0
  SET trackcount 0
  SET stickcount 0
  SET stick_group1 -1
  SET stick_group2 -1

  while (1)
    last_trackx := track_x
    last_tracky := track_y
 
    repeat 
      if (trackmask==COLORMASK_BLUE) && (Pink_Ball>0) then
        RETURN:GO_SEEBALL
      endif

      AP_GETCOLORARRAY colorarray
      AP_COLORFND colorarray trackmask track_x track_y track_level

      // If can't see track with current camera settings, try reevaluating...
      if (track_x<0) then
        if (trackmask==COLORMASK_BLUE) then
          if (!FSD_SEE_DISC()) then
            PLAY ACTION mad_eyes
            CALL G_WAITZERO
            RETURN:GO_IDLE
          endif
        else
          RETURN:GO_IDLE
        endif
      endif
    until (track_x>=0)

    // Detect sticking...
    if (F_ABS(track_x-last_trackx)<2) && (F_ABS(track_y-last_tracky)<2) then
      if ((F_ABS(xadjust)>=2) || (F_ABS(yadjust)>=2)) then
        if (trackcount==0) then
          trackarray[trackcount++] := track_x
          trackarray[trackcount++] := track_y
	else
          // Scan trackarray & see if current trackx/tracky combination has occured before...
          jj := -1
          for ii 0 trackcount 2        
            if (F_ABS(trackarray[ii]-track_x)<2) && (F_ABS(trackarray[ii+1]-track_y)<2) then
	      jj := ii
	    endif
	  next
	  if (jj>=0) && (jj != stick_group1) then
	    stickcount += (jj==stick_group2)
	    stick_group2 := stick_group1
	    stick_group1 := jj
	  endif
        endif
      endif
    endif

    xdelta := track_x-xtarget
    ydelta := track_y-ytarget
    xadjust := xdelta/3
    yadjust := ydelta/3

    // If not centered, then ensure head position adjusted by at least one tick...
    if (xadjust==0) then
      xadjust += (xdelta>2)-(xdelta<-2)
    endif
    if (yadjust==0) then
      yadjust += (ydelta>2)-(ydelta<-2)
    endif

    x_aligned := (xadjust==0) || (pan<-80) || (pan>80)
    y_aligned := (yadjust==0) || (tilt>=-15) || (tilt<-80)

    // Detect situations where the head sensor isn't moving (even though the head
    // might be bobbing slightly.  If this goes on too long, quit (probably close enough)...
    if (F_ABS(xadjust)<3) && (F_ABS(yadjust)<3) && ((Head_Pan==last_headpan) || (Head_Pan==last_headpan2)) then
      ++minor_count
      if (minor_count>6) then
	x_aligned := TRUE
	y_aligned := TRUE
      endif
    else
      // If make big adjustment, reset the minor tweak count...
      if (F_ABS(xadjust)>4) || (F_ABS(yadjust)>4) || (F_ABS(Head_Pan-last_headpan)>=8) then
        minor_count := 0
      endif
    endif

    // Head aligned on the disc? (or unable to move further towards it)
    if (x_aligned && y_aligned) || (stickcount>1) then
      if (F_ABS(xdelta)<4) then
        SET disc_pan Head_Pan-xdelta
      else
        SET disc_pan Head_Pan-xadjust
      endif

      if (F_ABS(ydelta)<4) then
        SET disc_tilt Head_Tilt-ydelta
      else
        SET disc_tilt Head_Tilt-yadjust
      endif

      PRINT "disc_pan=%d disc_tilt=%d left_lat=%d right_lat=%d" disc_pan disc_tilt LFLeg_Lat RFLeg_Lat
      RETURN:GO_AHEAD
    endif

    pan -= xadjust
    tilt -= yadjust
    if (tilt>-15) then
      tilt := -15
    endif

    PRINT "x/y_adjust=%d,%d   track_x,y=%d,%d  target_pan=%d" xadjust yadjust track_x track_y pan

    SET last_headpan2 last_headpan
    SET last_headpan Head_Pan

    SET init_wait Wait
    PLAY ACTION MOVE_HEAD pan tilt
    CALL G_WAIT
  wend
  RETURN:GO_IDLE



//////////////////////////////////////////////////////////////////////////////////
//
//  Attempt to relocate disc.  Called if loose sight of disc.  Different entry
//  points give a hint as to what as last attempted (movement wise) to aid in
//  the initial position guess.  If disc not found in hint location, then
//  do a wider scan for the disc.
//
//  Return:
//    GO_IDLE : Lost lock on the disc
//    GO_AHEAD : Disc centered (or as close as possibly centered)
//
:FSD_REAQUIRE_LEFT 	// Reaquire by looking left...
  aligned_disc := FALSE // can't be aligned if can't see disc
  PRINT "Reaquiring disc to left..."
  for i 1 3
    SET pan Head_Pan+10
    SET init_wait Wait
    PLAY ACTION MOVE.HEAD.FAST pan Head_Tilt
    CALL G_WAIT
    if (FSD_SEE_DISC()) then
      PRINT "  Found disc!"
      RETURN:GO_AHEAD
    endif
  next
  GO FSD_REAQUIRE_DISC


:FSD_REAQUIRE_RIGHT	// Reaquire by looking right...
  aligned_disc := FALSE // can't be aligned if can't see disc
  PRINT "Reaquiring disc to right..."
  for i 1 3
    SET pan Head_Pan-10
    SET init_wait Wait
    PLAY ACTION MOVE.HEAD.FAST pan Head_Tilt
    CALL G_WAIT
    if (FSD_SEE_DISC()) then
      PRINT "  Found disc!"
      RETURN:GO_AHEAD
    endif
  next
  GO FSD_REAQUIRE_DISC


:FSD_REAQUIRE_FORWARD	// Reaquire by looking forward...
  aligned_disc := FALSE // can't be aligned if can't see disc
  PRINT "Reaquiring disc forward..."
  for i 1 3
    SET tilt Head_Tilt+5
    SET init_wait Wait
    PLAY ACTION MOVE.HEAD.FAST Head_Pan tilt
    CALL G_WAIT
    if (FSD_SEE_DISC()) then
      PRINT "  Found disc!"
      RETURN:GO_AHEAD
    endif
  next
  GO FSD_REAQUIRE_DISC


:FSD_REAQUIRE_BACKWARD	// Reaquire by looking backward...
  aligned_disc := FALSE // can't be aligned if can't see disc
  PRINT "Reaquiring disc backward..."
  for i 1 3
    if (Head_Tilt<-70) then
      SET init_wait Wait
      PLAY ACTION WALK 180 20
      CALL G_WAIT
      if (FSD_SEE_DISC()) then
        PRINT "  Found disc!"
        RETURN:GO_AHEAD
      endif
      GO FSD_REAQUIRE_DISC
    endif

    SET tilt Head_Tilt-5
    SET init_wait Wait
    PLAY ACTION MOVE.HEAD.FAST Head_Pan tilt
    CALL G_WAIT
    if (FSD_SEE_DISC()) then
      PRINT "  Found disc!"
      RETURN:GO_AHEAD
    endif
  next
  GO FSD_REAQUIRE_DISC


//
// Didn't find disc in hint location (or no hint specified), so do full search.
// If not found, backup slightly & search again.  If still not found, give up.
//
:FSD_REAQUIRE_DISC
  aligned_disc := FALSE // can't be aligned if can't see disc
  PRINT "Searching to reaquire disc..."

  for i 1 10
    SET init_wait Wait
    SWITCH i
      CASE:1 PLAY ACTION MOVE.HEAD.FAST 50 -20	// look to right
      CASE:2 PLAY ACTION MOVE.HEAD.FAST 35 -45	// look to right
      CASE:3 PLAY ACTION MOVE.HEAD.FAST 0 -90	// look in front
      CASE:4 PLAY ACTION MOVE.HEAD.FAST -35 -45	// look to left
      CASE:5 PLAY ACTION MOVE.HEAD.FAST -50 -20	// look to left
      CASE:6 PLAY ACTION MOVE.HEAD.FAST 0 -40	// look in front
      CASE:7 PLAY ACTION WALK 180 60
      CASE:8 PLAY ACTION MOVE.HEAD.FAST 50 -20	// look to right
      CASE:9 PLAY ACTION MOVE.HEAD.FAST 35 -45	// look to right
      CASE:10 PLAY ACTION MOVE.HEAD.FAST 0 -90	// look in front
    CALL G_WAIT
    if (FSD_SEE_DISC()) then
      PRINT "  Found disc!"
      RETURN:GO_AHEAD
    endif
  next

  PRINT "  Disc not found..."
  RETURN:GO_IDLE


