// StikZap - main program
// version 1.3 open source version
// (c) 2006 CliePet

///////////////////////////////////////////////////////////////////////
// NOTE: this is a CLIE specific hack
//  will not work on other devices like WinCE/MobilePC PDAs
//    or USB PC memory stick readers
///////////////////////////////////////////////////////////////////////

#include <PalmOS.h>
#include <PCENativeCall.h>
#include "resource.h"

#include <Extensions/ExpansionMgr/ExpansionMgr.h>
#include <Extensions/ExpansionMgr/SlotDrvrLib.h>
#include <Extensions/ExpansionMgr/VFSMgr.h>

#include "endianutils.h"

#include "util.c_"
    // style note: program is one big source file
	// functional sub-systems are included in ".c_" source files
	// ".c_" files also used for data tables

///////////////////////////////////////////////////////////////////////
// Hardware magic (see README.TXT)

Int16 g_magicIndex = 0;
    // 0=>unknown, positive=> ARM, negative=> 68K

#include "fshelp.c_"
#include "types.c_"
#include "hwhelp.c_"
#include "guess.c_"

///////////////////////////////////////////////////////////////////////

static void DoCheck()
{
    UInt16 slotRef;
    UInt16 slotLibRef;

    // clear display
    SetLabel(MainInfo1, "(not tested)");
    SetLabel(MainInfo2, "(not tested)");
    SetLabel(MainInfo3, "");
    SetLabel(MainAdvice1, "Wait...");
    SetLabel(MainAdvice2, "");
    Redraw();

    ///// HARDWARE /////

    const char* szErr;
    if ((szErr = FindSlotInfo(slotRef, slotLibRef)) != NULL)
    {
        SetLabel(MainInfo1, szErr);
        SetLabel(MainAdvice1, "Insert stick and try again");
	    Redraw();
        return;
    }

    UInt8 hw_type[CB_HWCHECK];
    if (!ReadHwStickType(slotRef, slotLibRef, hw_type))
    {
        SetLabel(MainInfo1, "Unknown CLIE model");
        SetLabel(MainAdvice1, "Try GuessCpu command");
        SetLabel(MainAdvice2, "or email CliePet");
	    Redraw();
        return;
    }
	const STIK_TYPE* pHwType = NULL;
	const char* szHwType = FindHwType(hw_type, pHwType);
    if (pHwType == NULL)
    {
	    if (szHwType != NULL)
	        szHwType = "blank"; // usually after reformat
        else
	        szHwType = "unknown";
    }
    SetLabel(MainInfo1, szHwType);

    ///// SOFTWARE - MSWATCH /////

    const char* szAdvice = NULL; // best advice so far
    const char* szAdvice2 = NULL;

    UInt8 sw_type[CB_SWCHECK];
    bool bSwTypeValid = false;
    if (!FS_Init_mstk())
    {
        SetLabel(MainInfo2, "Memory stick not mounted");
        SetLabel(MainInfo3, "Stick probably corrupt");
        szAdvice = "Reformat the stick";
    }
    else
    {
		FileRef pf = FS_open(PATH_SWCHECK, "r");
        if (pf != NULL)
        {
            if (FS_read(pf, sw_type, CB_SWCHECK))
                bSwTypeValid = true;
            FS_close(pf);
        }
        if (!bSwTypeValid)
        {
            SetLabel(MainInfo2, "No AiboWare detected");
	        szAdvice = "Restore full stick backup";
        }
    }

	const STIK_TYPE* pSwType = NULL;
    if (bSwTypeValid)
    {
		const char* szSwType = FindSwType(sw_type, pSwType);
        if (szSwType != NULL)
	        SetLabel(MainInfo2, szSwType);
        else
	        SetLabel(MainInfo2, "unknown");
    }

    ///// SOFTWARE - SAFETY FILE /////
        // normal file (hopefully) containing the same 1K as the hardware key
    UInt8 bk_type[CB_HWCHECK];
    bool bSafeFileValid = false;
	const STIK_TYPE* pSafeFileType = NULL;
    if (bSwTypeValid)
    {
		FileRef pf = FS_open(PATH_SAFETYFILE, "r");
        if (pf != NULL)
        {
            if (FS_read(pf, bk_type, CB_HWCHECK))
                bSafeFileValid = true;
            FS_close(pf);
        }
        if (bSafeFileValid)
			FindHwType(bk_type, pSafeFileType);
    }

    ///// ADVICE PT1 - get a valid safety /////
    if (!bSafeFileValid || pSafeFileType == NULL)
    {
	    if (!bSafeFileValid)
	        SetLabel(MainInfo3, "no safety file");
        else
	        SetLabel(MainInfo3, "unknown safety file");
        if (szAdvice == NULL)
        {
	        szAdvice = "Create Safety";
	        szAdvice2 = "then copy entire stick to PC";
        }
    }
    else if (pSafeFileType == pSwType && pSafeFileType == pHwType)
    {
        SetLabel(MainInfo3, "safety file matches both");
        // nothing wrong
    }
    else if (pSafeFileType == pSwType) // && pSafeFileType != pHwType
    {
        SetLabel(MainInfo3, "safety file matches software");
        // typically right before restore
    }
    else if (pSafeFileType == pHwType) // && pSafeFileType != pSwType
    {
        SetLabel(MainInfo3, "safety file is wrong");
        if (szAdvice == NULL)
	        szAdvice = "Restore full stick backup";
    }
    else // valid safety, but pSafeFileType matches neither
    {
        SetLabel(MainInfo3, "safety file is totally wrong");
        if (szAdvice == NULL)
        {
	        szAdvice = "Create Safety";
	        szAdvice2 = "then copy entire stick to PC";
        }
    }

    ///// ADVICE PT2 - check sw and hw /////
    if (szAdvice == NULL && pHwType != NULL && pHwType == pSwType)
    {
        szAdvice = "Everything OK";
    }
    else if (szAdvice == NULL && pSwType != NULL && pSafeFileType != NULL &&
      pSafeFileType == pSwType && pHwType != pSwType)
    {
        szAdvice = "Press 'Use Safety'";
        if (pHwType != NULL)
	        szAdvice2 = "(only if you kept a backup)";
    }

    if (pHwType == NULL && pSwType != NULL)
    {
        // overrides other results
        AlertNotify("Please run 'Create Safety' command\nand email the file\n"
			PATH_SAFETYFILE "\n"
            " to AiboPet");
        szAdvice = "Create Safety, then email the file";
    }

    ///// SHOW ADVICE /////
    if (szAdvice == NULL)
        szAdvice = "No good advice";
	SetLabel(MainAdvice1, szAdvice);
    if (szAdvice2 != NULL)
		SetLabel(MainAdvice2, szAdvice2);
    Redraw();
}

static void DoCreateSafetyFile()
{
    UInt16 slotRef;
    UInt16 slotLibRef;
    const char* szErr = FindSlotInfo(slotRef, slotLibRef);
    if (szErr != NULL)
    {
    	AlertError(szErr);
        return;
    }

    UInt8 hw_type[CB_HWCHECK];
    if (!ReadHwStickType(slotRef, slotLibRef, hw_type))
    {
        AlertError("Unknown CLIE model");
        return;
    }

    if (!FS_Init_mstk())
    {
        AlertError("Memory stick not mounted\nStick probably corrupt");
        return;
    }
	FileRef pf = FS_open(PATH_SAFETYFILE, "w");
    if (pf == NULL)
    {
        AlertError("Can't create safety file\n"
			PATH_SAFETYFILE "\n"
	        "/OPEN-R folder may be missing (if no software)\n"
	        "or stick may be write protected");
        return;
    }
    bool bOK = true;
	if (!FS_write(pf, hw_type, CB_HWCHECK))
        bOK = false;

    if (!FS_close(pf))
        bOK = false;

    if (bOK)
        AlertNotify("Hardware key saved\n"
        "in safety file\n"
		PATH_SAFETYFILE "\n"
        "Backup entire stick now\n" /* max length line */
        "Save on your PC");
    else
        AlertError("Can't write safety file\n"
		PATH_SAFETYFILE "\n"
        "Stick may be full");
}

static void DoZap(const UInt8* new_type)
{
    UInt16 slotRef;
    UInt16 slotLibRef;
    const char* szErr = FindSlotInfo(slotRef, slotLibRef);
    if (szErr != NULL)
    {
    	AlertError(szErr);
        return;
    }

    UInt8 old_type[CB_HWCHECK];
    if (!ReadHwStickType(slotRef, slotLibRef, old_type))
    {
        AlertError("Unknown CLIE model");
        return;
    }

    if (!AskWriteConfirm(old_type, new_type))
    {
    	AlertNotify("Write Aborted");
        return;
    }

	PerformWriteVerbose(new_type);
}


static void DoUseSafetyFile()
{
    UInt16 slotRef;
    UInt16 slotLibRef;
    const char* szErr = FindSlotInfo(slotRef, slotLibRef);
    if (szErr != NULL)
    {
    	AlertError(szErr);
        return;
    }

    UInt8 old_type[CB_HWCHECK];
    UInt8 new_type[CB_HWCHECK];
    if (!ReadHwStickType(slotRef, slotLibRef, old_type))
    {
        AlertError("Unknown CLIE model");
        return;
    }

    if (!FS_Init_mstk())
    {
        AlertError("Memory stick not mounted\nStick probably corrupt");
        return;
    }
	FileRef pf = FS_open(PATH_SAFETYFILE, "r");
    if (pf == NULL)
    {
        AlertError("Can't find safety file\n"
			PATH_SAFETYFILE "\n"
            "Unable to restore");
        return;
    }
    bool bOK = true;
	if (!FS_read(pf, new_type, CB_HWCHECK))
        bOK = false;
    if (!FS_close(pf))
        bOK = false;

    if (!bOK)
    {
        AlertError("Can't read safety file\n"
			PATH_SAFETYFILE "\n"
            "Unable to restore");
        return;
    }

    if (!AskWriteConfirm(old_type, new_type))
    {
    	AlertNotify("Write Aborted");
        return;
    }

	PerformWriteVerbose(new_type);
}

///////////////////////////////////////////////////////////////

static void FormatTag(char szT[5], UInt32 l)
{
    MemMove(&szT[0], &l, 4);
    szT[4] = '\0';
}

static void DoCpuGuess()
{
    UInt32 halid, devid;
	UInt32 processor;
	if (FtrGet(sysFtrCreator, sysFtrNumOEMHALID, &halid) != 0 ||
	    FtrGet(sysFtrCreator, sysFtrNumOEMDeviceID, &devid) != 0 ||
		FtrGet(sysFtrCreator, sysFtrNumProcessorID, &processor) != 0)
    {
        AlertError("Can't figure out CPU or device type");
        return;
    }

    bool bIsArm = sysFtrNumProcessorIsARM(processor);
    int indexGuess = bIsArm ? CpuGuessArm(true) : CpuGuess68K(true);
	    
	if (indexGuess != 0)
    {
	    char sz1[5];
	    FormatTag(sz1, devid);
	    char sz2[5];
	    FormatTag(sz2, halid);
	    char szReport[128];
	    StrPrintF(szReport, "CPU INFO:\n'%s %s %d'", sz1, sz2, indexGuess);
	    AlertNotify(szReport);
    }
    else
    {
        AlertError("Guessing failed");
    }
}

static void DoCpuSave()
{
    UInt16 slotRef;
    UInt16 slotLibRef;
    const char* szErr = FindSlotInfo(slotRef, slotLibRef);
    if (szErr != NULL)
    {
    	AlertError(szErr);
        return;
    }
	UInt32 processor;
	if (FtrGet(sysFtrCreator, sysFtrNumProcessorID, &processor) != 0)
    {
        AlertError("Error - get cpu type");
        return;
    }
    bool bIsArm = sysFtrNumProcessorIsARM(processor);
    UInt32* plSave = NULL; // saving internal to same memstick - should be ok
    if (bIsArm)
    {
	    UInt32 results[1];
	    if (!CallArmlet(0, results))
	    {
	    	AlertError("bad CLIE model");
	        return;
	    }
	    plSave = (UInt32*)results[0];
    }
    else
    {
		SysLibTblEntryPtr libEntPtr = SysLibTblEntry(slotLibRef);
		UInt32* plGlob = (UInt32*)(libEntPtr->globalsP);
	    if (plGlob == NULL)
        {
            AlertError("Error - get 68k");
	        return;
        }
        plSave = plGlob;
	}
    int cbSave = MemPtrSize((MemPtr)plSave);
    if (cbSave <= 0)
    {
        AlertError("Block size error");
        return;
    }
#if 0
    char szT[128];
    StrPrintF(szT, "%08lX %08lX", (UInt32)&plSave, (UInt32)plSave);
    AlertNotify(szT);
#endif
    
    if (!FS_Init_mstk())
    {
        AlertError("No memory stick to save to");
        return;
    }
	FileRef pf = FS_open("/INTERNAL.BIN", "w");
    if (pf == NULL)
    {
        AlertError("CREATE ERROR");
        return;
    }
	if (!FS_write(pf, plSave, cbSave))
        AlertError("WRITE ERROR");
	FS_close(pf);
    AlertNotify("/INTERNAL.BIN saved on stick");
}

/////////////////////////////////////////////////////////////

static Boolean MainFormDoCommand(UInt16 command)
{
	switch (command)
	{
	case MainOptionsAboutStarterApp:
		MenuEraseStatus(0);
        {
			FormPtr frmP = FrmInitForm (AboutForm);
			FrmDoDialog (frmP);
			FrmDeleteForm (frmP);
        }
		break;
	case MainOptionsCheckType:
		DoCheck();
		break;
    case MainOptionsAutoFix:
        AlertNotify("Not in this version");
        break;

	case MainOptionsMakePMS:
		DoZap(g_type_PMS);
		break;
    case MainOptionsMakeERS7:
		DoZap(g_type_ERS7);
		break;
	case MainOptionsCpuGuess:
        DoCpuGuess();
        break;
    case MainOptionsSaveCpu:
		DoCpuSave();
        break;
    case MainOptionsCreateSafety:
		DoCreateSafetyFile();
        break;
    case MainOptionsUseSafety:
		DoUseSafetyFile();
        break;

    default: // not handled
		return false;
	}
	
	return true;    // handled
}

static Boolean MainFormHandleEvent(EventPtr eventP)
{
	Boolean handled = false;
	FormPtr frmP;

	switch (eventP->eType) 
		{
		case menuEvent:
			return MainFormDoCommand(eventP->data.menu.itemID);

        case ctlSelectEvent:
            return MainFormDoCommand(eventP->data.ctlSelect.controlID);

		case frmOpenEvent:
			frmP = FrmGetActiveForm();
			//MainFormInit( frmP);
            SetLabel(MainInfo2, "");
            SetLabel(MainAdvice2, "");
			FrmDrawForm(frmP);
			DoCheck();
			handled = true;
			break;
			
		case frmUpdateEvent:
			break;

		default:
			break;
		}
	
	return handled;
}


static Boolean AppHandleEvent(EventPtr eventP)
{
	if (eventP->eType == frmLoadEvent)
	{
		UInt16 formId = eventP->data.frmLoad.formID;
		FormPtr frmP = FrmInitForm(formId);
		FrmSetActiveForm(frmP);
		if (formId == MainForm)
			FrmSetEventHandler(frmP, MainFormHandleEvent);
		return true;
	}
	return false;
}

static void AppEventLoop(void)
{
	EventType event;
	do
    {
		EvtGetEvent(&event, evtWaitForever);
		UInt16 error;

		if (!SysHandleEvent(&event))
			if (!MenuHandleEvent(0, &event, &error))
				if (!AppHandleEvent(&event))
					FrmDispatchEvent(&event);

	} while (event.eType != appStopEvent);
}


#include "SonyOEM.h"
struct CPUINFO
{
    UInt32 halid;
    UInt32 devid;
    UInt16 magicIndex;
};

// Tested versions
static const CPUINFO g_cpuInfo[] = 
{
    // Tested and confirmed
    { 'ancy', 'ancy', 0x6E0/4 }, // VZ90 - like TH55
    { sonyHwrOEMHALID_TH55, sonyHwrOEMDeviceID_TH55, 0x6E0/4 },
    { sonyHwrOEMHALID_UX50, sonyHwrOEMDeviceID_UX50, 0x460/4 },
    { sonyHwrOEMHALID_NZ90, sonyHwrOEMDeviceID_NZ90, 0x1D4/4 },
    { sonyHwrOEMHALID_NR70, sonyHwrOEMDeviceID_NR70, -(0x1C/4) },
    { sonyHwrOEMHALID_SL10, sonyHwrOEMDeviceID_SL10, -5 }, // with smarter guess

    { sonyHwrOEMHALID_NX80, sonyHwrOEMDeviceID_NX80, 0x1D8/4 },
    { sonyHwrOEMHALID_TJ35, sonyHwrOEMDeviceID_TJ35, 0x1C4/4 },
    { sonyHwrOEMHALID_TJ25, sonyHwrOEMDeviceID_TJ25, 0x1C4/4 },
    { sonyHwrOEMHALID_UX40, sonyHwrOEMDeviceID_UX40, 0x45C/4 },
    { sonyHwrOEMHALID_NX70, sonyHwrOEMDeviceID_NX70, 0x1D0/4 },
};
#define NUM_KNOWN_CPUS (sizeof(g_cpuInfo)/sizeof(CPUINFO))


UInt32 PilotMain( UInt16 cmd, MemPtr cmdPBP, UInt16 launchFlags)
{
    if (cmd == sysAppLaunchCmdNotify)
    {
        // low level system notification
        SysNotifyParamType* psn = (SysNotifyParamType*)cmdPBP;
		if (psn->notifyType == sysNotifyVolumeUnmountedEvent)
        {
            // exit the app when someone removes the stick
			EventType et;
			MemSet(&et, sizeof(et), 0);
			et.eType = appStopEvent;
		    EvtAddEventToQueue(&et);
        }
        return 0;
    }

	if (cmd != sysAppLaunchCmdNormalLaunch)
		return errNone;

    UInt32 halid, devid;
	UInt32 processor;
	if (FtrGet(sysFtrCreator, sysFtrNumOEMHALID, &halid) != 0 ||
	    FtrGet(sysFtrCreator, sysFtrNumOEMDeviceID, &devid) != 0 ||
		FtrGet(sysFtrCreator, sysFtrNumProcessorID, &processor) != 0)
    {
        AlertError("Can't figure out CPU or device type");
        return 1;
    }

    bool bIsArm = sysFtrNumProcessorIsARM(processor);

    // BLOCK: Identify
    {
        int iSameHal = -1;
        int iSameDev = -1;
	    for (int iCPU = 0; iCPU < NUM_KNOWN_CPUS; iCPU++)
	    {
            CPUINFO const& ci = g_cpuInfo[iCPU];
            if (halid == ci.halid && iSameHal == -1)
                iSameHal = iCPU;
            if (devid == ci.devid && iSameDev == -1)
                iSameDev = iCPU;
	    }

        if (iSameDev != -1)
        {
			g_magicIndex = g_cpuInfo[iSameDev].magicIndex;
        }
        else if (iSameHal != -1)
        {
            AlertNotify("Untested device, assuming similar to known one");
			g_magicIndex = g_cpuInfo[iSameHal].magicIndex;
        }

        if (g_magicIndex == 0)
        {
            AlertNotify("Unknown device, program will guess");
        }
    }
	if (g_magicIndex == -1)
    {
        AlertError("Support for your model CLIE is not complete");
        return 1;
    }

    if (bIsArm)
    {
	    int indexArmGuess = CpuGuessArm(false);
        if (indexArmGuess > 0 && g_magicIndex > 0 && indexArmGuess != g_magicIndex)
        {
            AlertError("WARNING: known device, but guess was different");
            g_magicIndex = 0;
        }

        if (indexArmGuess > 0 && g_magicIndex == 0)
        {
            char szT[64];
            StrPrintF(szT, "Using CPU guess %d\nIf it works, run CpuGuess command and email CliePet", indexArmGuess);
            AlertNotify(szT);
            g_magicIndex = indexArmGuess;
        }
    }
    else
    {
	    int index68KGuess = CpuGuess68K(false);
        if (index68KGuess < 0 && g_magicIndex < 0 && index68KGuess != g_magicIndex)
        {
            AlertError("WARNING: known device\nBut guess was different");
            g_magicIndex = 0;
        }

        if (index68KGuess < 0 && g_magicIndex == 0)
        {
            char szT[64];
            StrPrintF(szT, "Using CPU guess %d\nIf it works, run CpuGuess command and email CliePet", index68KGuess);
            AlertNotify(szT);
            g_magicIndex = index68KGuess;
        }
    }

    if (g_magicIndex == 0)
    {
#if 0
        AlertError("Unknown device, guessing didn't help\nprogram will not run");
        return 1;
#else
        AlertError("Unknown device, guessing didn't help\nUse GuessCpu or SaveCpu only");
        // continue running
#endif
    }
        
    if ((g_magicIndex > 0 && !bIsArm) || (g_magicIndex < 0 && bIsArm))
    {
        AlertError("Internal Error - ARM/68K mismatch");
        return 1;
    }

	UInt16 cardNo;
	LocalID dbid;
	SysCurAppDatabase(&cardNo, &dbid);

	SysNotifyRegister(cardNo, dbid, sysNotifyVolumeUnmountedEvent,
		 NULL, sysNotifyNormalPriority, NULL);

	FrmGotoForm(MainForm);
	AppEventLoop();

	FrmCloseAllForms();
	SysNotifyUnregister(cardNo, dbid, sysNotifyVolumeUnmountedEvent,
		 sysNotifyNormalPriority);

	return errNone;
}


